import enum
import logging
import pytils
import pytz

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.pusher import BasePusher
from cars.core.sms import SmsNotifier
from cars.core.sender import EmailNotifier
from cars.settings import FINES as settings

LOGGER = logging.getLogger(__name__)


class FineNotifier(object):
    USER_FINE_TAG = 'pdd_fine_push'

    class SmsMessageCode(enum.Enum):
        FINE_CHARGED = 'fine.charged.success'

    class SenderCampaign(enum.Enum):
        FINE_CHARGED = 'fine_charge_notification2'
        TWO_WEEKS_OLDER = '4P7IH8I3-941'

    def __init__(self, saas_client, push_notifier, sms_notifier, email_notifier):
        self._saas_client = saas_client
        self._push = push_notifier
        self._sms = sms_notifier
        self._email = email_notifier

    @classmethod
    def from_settings(cls):
        return cls(
            saas_client=SaasDriveAdminClient.from_settings(),
            push_notifier=BasePusher.from_settings(),
            sms_notifier=SmsNotifier.from_settings(),
            email_notifier=EmailNotifier.from_settings(),
        )

    def _push_notify_fine_charged_deprecated(self, fine):
        text = 'Нарушение ПДД. Штраф — {} рублей. Подробности в почте.'.format(
            int(fine.sum_to_pay_without_discount),
        )
        self._push.send(uid=fine.user.uid, message=text)

    def push_notify_fine_charged(self, fine):
        comment = '#{}'.format(fine.ruling_number)
        self._saas_client.add_user_tag(
            user_id=fine.user.id, tag=self.USER_FINE_TAG, comment=comment, re_raise=True
        )

    def sms_notify_fine_charged(self, fine):
        text = 'Вы нарушили правило дорожного движения за рулём Яндекс.Драйва. С вашей карты спишут {sum_to_pay} рублей. Подробности в почте.'.format(
            sum_to_pay=int(fine.sum_to_pay),
            sum_to_pay_without_discount=int(fine.sum_to_pay_without_discount),
        )
        self._sms.send(phone=fine.user.phone.as_e164, text=text)

    def email_notify_fine_charged(self, fine, campaign_name=None):
        if fine.user.email is None:
            LOGGER.info(
                'not found email for fine_id=%s user_id=%s',
                fine.id,
                fine.user.id,
            )
            return

        campaign = getattr(self.SenderCampaign, campaign_name or 'FINE_CHARGED').value

        if fine.odps_name == 'МАДИ':
            order_author = 'МАДИ'
        else:
            order_author = 'ГИБДД'

        paragraph = '{}. Постановление {}'.format(
            fine.article_koap,
            fine.odps_name,
        )

        strike_date = pytils.dt.ru_strftime(
            "%d %B в %H:%M",
            inflected=True,
            date=fine.violation_time.astimezone(
                pytz.timezone('Europe/Moscow')
            ),
        )

        # won't work for Major fines
        fine_article_title = fine.article_koap.split(' - ')[0]
        if fine_article_title in settings['koap_no_discount']:
            discount_string = 'К сожалению, по этому типу штрафов скидка на оплату не предоставляется. Мы оплатили штраф'
        else:
            discount_string = 'Мы оплатили штраф со скидкой 50%'

        autocode_url = 'https://mos.ru/shtrafy/result?dap={}'.format(fine.ruling_number)

        args = {
            'car_plate': fine.car.number,
            'car_model': fine.car.model.name,
            'violation_place': fine.violation_place,
            'autocode_url': autocode_url,
            'ruling_number': fine.ruling_number,
            'strike_date': strike_date,
            'paragraph': paragraph,
            'order_author': order_author,  # may be not needed
            'sum': int(fine.sum_to_pay_without_discount),
            'final_sum': int(fine.sum_to_pay),
            'discount_string': discount_string,  # may be not needed
        }

        photos = list(fine.photos.all())
        if photos:
            args['photo'] = photos[0].url  # may be not needed

            template = '<span style="color:#949494">{title}</span><br>{photo_tags}<br>'

            title = 'Изображения с камеры' if len(photos) > 1 else 'Изображение с камеры'
            photo_tags = [
                '<img src="{}" style="max-width:571px; padding-top: 7px;" alt="" border="0"><br>'.format(photo.url)
                for photo in photos
            ]

            photo_code_block = template.format(title=title, photo_tags=''.join(photo_tags))
            args['photo_code_block'] = photo_code_block

        self._email.send(
            campaign=campaign,
            to_email=fine.user.email,
            args=args,
            assert_ok=True
        )
