import datetime
from decimal import Decimal
import contextlib
import logging
import pytz
import re
import time
import socket

from django.db.models import Count, Q
from django.db import transaction
from django.utils import timezone

from cars.carsharing.models import Car
from cars.core.autocode import AutoCodeFinesGetter
from cars.core.util import datetime_helper
from cars.core.saas_drive import SaasDrive
from cars.settings import FINES as settings
from cars.users.models.user import User

from ..models import AutocodeFine, AutocodeFinePhoto
from .fine_collector import FineCollector
from .fine_photos_manager import FinePhotosManager

from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin

LOGGER = logging.getLogger(__name__)


def _datetime_to_timestamp(dt):
    return int(
        (
            dt -
            timezone.make_aware(
                datetime.datetime(1970, 1, 1),
                pytz.utc,
            )
        ).total_seconds()
    )


def _parse_date(date_str):
    if not date_str:
        return None
    try:
        parsed_date = datetime.datetime.strptime(date_str, '%d.%m.%Y')
        parsed_date = timezone.make_aware(parsed_date, timezone=pytz.UTC)
        return parsed_date
    except ValueError:
        LOGGER.exception('Date parsing failed: %s', date_str)
        return None


def _parse_date_and_time(datetime_str):
    try:
        parsed_datetime = datetime.datetime.strptime(datetime_str, '%d.%m.%Y %H:%M:%S')
        return timezone.make_aware(
            parsed_datetime,
            pytz.timezone('Europe/Moscow'),
        )
    except ValueError:
        LOGGER.exception('DateTime parsing failed: %s', datetime_str)
        return None


class FinesManager(object):
    SKIPPED_LIMIT = 4

    def __init__(self, *, autocode_client, fine_collector, photo_manager, saas_client):
        self._autocode_client = autocode_client
        self._fine_collector = fine_collector
        self._photo_manager = photo_manager
        self._saas_client = saas_client

        self._default_since = datetime_helper.utc_localize(datetime.datetime(2019, 12, 1, 0, 0, 0))

    @classmethod
    def from_settings(cls):
        return cls(
            autocode_client=AutoCodeFinesGetter.from_settings(),
            fine_collector=FineCollector.from_settings(),
            photo_manager=FinePhotosManager.from_settings(),
            saas_client=SaasDrive.from_settings(
                url='https://prestable.carsharing.yandex.net',
                version='v5.0.0',
                prestable=False,
            )
        )

    def _get_sync_entry(self, sync_origin):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=sync_origin,
                    last_data_sync_time=self._default_since,
                    active_until=self._default_since,
                )

            if sync_entry.data is None:
                sync_entry.data = {}
                sync_entry.save()

        return sync_entry

    def _get_sync_entry_for_update(self, sync_origin):
        sync_entry = CallStatSyncStatus.objects.select_for_update().get(origin=sync_origin)
        return sync_entry

    def check_is_active(self, *, sync_origin=None, sync_entry=None):
        if sync_entry is None:
            sync_entry = self._get_sync_entry(sync_origin)

        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now

        is_active = started and not finished
        return is_active

    def attach_missing_photos(self):
        sync_entry = self._get_sync_entry(sync_origin=self.fine_photo_collecting_sync_origin)
        if not self.check_is_active(sync_entry=sync_entry):
            return

        # use explicit filtering to determine fines without photo
        photo_fine_ids = set(AutocodeFinePhoto.objects.all().values_list('fine_id', flat=True))

        fines = AutocodeFine.objects.filter(has_photo=True).values('id', 'ruling_number')

        for fine in fines:
            if fine['id'] in photo_fine_ids:
                continue

            try:
                photos_raw = self._autocode_client.get_violation_photos(fine['ruling_number'])
            except Exception:
                LOGGER.exception('Failed to get violation photo, ruling number=%s', fine['ruling_number'])
                continue  # to be collected the next time

            for fine_b64content in photos_raw:
                try:
                    self._photo_manager.add_fine_photo(fine['id'], fine_b64content)
                except Exception:
                    LOGGER.exception('Unable to save fine photo, ruling number=%s', fine['ruling_number'])

    def subscribe_all_cars(self):
        """
        Subscribe all cars that are currently present in the DB table.
        Using STS numbers.
        """
        stses = [
            str(x) for x in Car.objects.values_list('registration_id', flat=True)
            if x  # sometimes car doesn't have registration_id yet
        ]
        self._autocode_client.subscribe_vehicles(stses)
        LOGGER.info('%d car(s) had been successfully subscribed', len(stses))

    def collect_new_fines(self):
        """
        Collect all new fine notifications.
        """
        sync_entry = self._get_sync_entry(sync_origin=self.fine_collecting_sync_origin)
        if not self.check_is_active(sync_entry=sync_entry):
            return []

        new_fines = self._autocode_client.get_new_fines(limit=10**6)

        new_fines_db_objects = []

        for fine in new_fines:
            time.sleep(0.05)
            try:
                new_fines_db_objects.append(self._create_new_fine_db_object(fine))
            except Exception:
                LOGGER.exception('Exception while creating fine DB object for fine `{}`'.format(fine))

        LOGGER.info(
            '%d new fine(s) had been successfully received from autocode.',
            len(new_fines_db_objects)
        )

        return new_fines_db_objects

    @transaction.atomic(savepoint=False)
    def save_new_fines(self, raw_fines):
        save_duplicates_from_other_sources = False
        duplicates = []

        already_present_ruling_numbers = set(
            AutocodeFine.objects
            .filter(
                ruling_number__in=[f.ruling_number for f in raw_fines]
            )
            .values_list('ruling_number', flat=True)
        )

        added_fines_number = 0

        fines = []
        for fine in raw_fines:
            if fine.ruling_number not in already_present_ruling_numbers:
                fines.append(fine)
                fine.save()
                already_present_ruling_numbers.add(fine.ruling_number)
                added_fines_number += 1
            else:
                duplicates.append(fine)

        LOGGER.info('{} fine(s) had been successfully added into the DB'.format(added_fines_number))

        if duplicates:
            LOGGER.info('Processing {} fines with duplicated ruling numbers'.format(len(duplicates)))

            for duplicated_fine in duplicates:
                processed_fine = self._process_fine_duplicate(duplicated_fine, save_duplicates_from_other_sources)
                if processed_fine is not None:
                    fines.append(processed_fine)

        return fines, added_fines_number

    def _process_fine_duplicate(self, duplicated_fine, save_duplicates_from_other_sources):
        original_ruling_number = duplicated_fine.ruling_number

        similar_fine_exists = AutocodeFine.objects.filter(
            ruling_number=original_ruling_number, source_type=duplicated_fine.source_type
        ).exists()

        processed_fine = None

        if not similar_fine_exists:
            if not save_duplicates_from_other_sources:
                LOGGER.info(
                    'There is a fine duplicate from other source with ruling number - {}; to be processed further'
                    .format(original_ruling_number)
                )
            else:
                LOGGER.info(
                    'Saving fine duplicate from other source: ruling number - {}'
                    .format(original_ruling_number)
                )

                duplicated_fine.needs_charge = False
                duplicated_fine.save()
                processed_fine = duplicated_fine
        else:
            LOGGER.info('No need to save fine duplicate: ruling number - {}'.format(original_ruling_number))

        return processed_fine

    @property
    def sync_origin(self):
        return SyncOrigin.FINES_CHARGING.value

    @property
    def fine_payments_sync_origin(self):
        return SyncOrigin.FINES_PAYMENTS_CHECK.value

    @property
    def fine_collecting_sync_origin(self):
        return SyncOrigin.FINES_COLLECTING.value

    @property
    def fine_photo_collecting_sync_origin(self):
        return SyncOrigin.FINE_PHOTO_COLLECTING.value

    @property
    def process_unique_shard_id(self):
        return socket.getfqdn()

    @property
    def lock_timeout(self):
        return datetime.timedelta(hours=3)

    def _is_locked(self, sync_entry, lock_timeout=None):
        if sync_entry.data.get('locked', None) is not None:
            return lock_timeout is None or datetime_helper.now() < sync_entry.last_data_sync_time + lock_timeout
        return False

    def _is_lock_acquired(self, sync_entry, lock_timeout=None):
        return (
                self._is_locked(sync_entry, lock_timeout) and
                sync_entry.data['locked'] == self.process_unique_shard_id
        )

    def _acquire_lock(self, sync_origin, lock_timeout=None):
        with transaction.atomic(savepoint=False):
            sync_entry = self._get_sync_entry_for_update(sync_origin=sync_origin)

            if not self._is_locked(sync_entry, lock_timeout):
                sync_entry.last_data_sync_time = datetime_helper.now()
                sync_entry.data['locked'] = self.process_unique_shard_id
                sync_entry.save()

        sync_entry = self._get_sync_entry(sync_origin=sync_origin)
        return self._is_lock_acquired(sync_entry, lock_timeout)

    def _release_lock(self, sync_origin, lock_timeout=None):
        sync_entry = self._get_sync_entry(sync_origin=sync_origin)
        if self._is_lock_acquired(sync_entry, lock_timeout):
            with transaction.atomic(savepoint=False):
                sync_entry = self._get_sync_entry_for_update(sync_origin=sync_origin)
                sync_entry.data['locked'] = None
                sync_entry.save()

    @contextlib.contextmanager
    def _lock_releasing(self, sync_origin, lock_timeout=None):
        try:
            yield
        finally:
            self._release_lock(sync_origin, lock_timeout)

    def process_existing_fines(self):
        sync_entry = self._get_sync_entry(sync_origin=self.sync_origin)
        if not self.check_is_active(sync_entry=sync_entry):
            return

        lock_timeout = self.lock_timeout

        if not self._acquire_lock(self.sync_origin, lock_timeout):
            LOGGER.error('cannot acquire lock')
            return

        with self._lock_releasing(self.sync_origin, lock_timeout):
            rebind_required_since_ts = sync_entry.data.get('rebind_required_since_ts', None)
            if rebind_required_since_ts is not None:
                self.rebind_fines(rebind_required_since_ts)

                with transaction.atomic(savepoint=False):
                    sync_entry = self._get_sync_entry_for_update(sync_origin=self.sync_origin)
                    sync_entry.data['rebind_required_since_ts'] = None
                    sync_entry.save()

            collect_kwargs = self._get_collect_kwargs(sync_entry)

            query = self._fine_collector.get_collectable_query(**collect_kwargs)
            qs = AutocodeFine.objects.filter(query)
            limit = sync_entry.data.get('limit', None)

            email_campaign_name = sync_entry.data.get('email_campaign_name', None)
            sum_limit_per_day = sync_entry.data.get('sum_limit_per_day', None)
            count_limit_per_day = sync_entry.data.get('count_limit_per_day', None)

            for fine in qs:
                updated = self._fine_collector.collect(fine, email_campaign_name, sum_limit_per_day, count_limit_per_day)

                if limit is not None:
                    limit -= int(updated)
                    if limit <= 0:
                        break

                time.sleep(0.05)

    def _get_collect_kwargs(self, sync_entry):
        charge_limit = sync_entry.data.get('charge_limit', None)
        charge_time_limit_days = sync_entry.data.get('charge_time_limit_days', None)
        charge_time_limit = datetime.timedelta(days=charge_time_limit_days) if charge_time_limit_days is not None else None
        source_type = sync_entry.data.get('source_type', None)
        article_koap = sync_entry.data.get('article_koap', None)
        article_koap_like = sync_entry.data.get('article_koap_like', None)
        ruling_date_since_ts = sync_entry.data.get('ruling_date_since_ts', None)
        ruling_date_since = datetime_helper.timestamp_to_datetime(ruling_date_since_ts) if ruling_date_since_ts is not None else None
        ruling_date_until_ts = sync_entry.data.get('ruling_date_until_ts', None)
        ruling_date_until = datetime_helper.timestamp_to_datetime(ruling_date_until_ts) if ruling_date_until_ts is not None else None
        info_received_since_ts = sync_entry.data.get('info_received_since_ts', None)
        info_received_since = datetime_helper.timestamp_to_datetime(info_received_since_ts) if info_received_since_ts is not None else None
        info_received_until_ts = sync_entry.data.get('info_received_until_ts', None)
        info_received_until = datetime_helper.timestamp_to_datetime(info_received_until_ts) if info_received_until_ts is not None else None
        violation_since_ts = sync_entry.data.get('violation_since_ts', None)
        violation_since = datetime_helper.timestamp_to_datetime(violation_since_ts) if violation_since_ts is not None else None
        violation_until_ts = sync_entry.data.get('violation_until_ts', None)
        violation_until = datetime_helper.timestamp_to_datetime(violation_until_ts) if violation_until_ts is not None else None
        article_koap_not_like = sync_entry.data.get('article_koap_not_like', None)
        has_discount = sync_entry.data.get('has_discount', None)
        collect_kwargs = dict(
            charge_limit=charge_limit,
            charge_time_limit=charge_time_limit,
            source_type=source_type,
            article_koap=article_koap,
            article_koap_like=article_koap_like,
            article_koap_not_like=article_koap_not_like,
            ruling_date_since=ruling_date_since,
            ruling_date_until=ruling_date_until,
            info_received_since=info_received_since,
            info_received_until=info_received_until,
            violation_since=violation_since,
            violation_until=violation_until,
            has_discount=has_discount
        )
        return collect_kwargs

    def process_fine_payments(self):
        sync_entry = self._get_sync_entry(sync_origin=self.fine_payments_sync_origin)
        if not self.check_is_active(sync_entry=sync_entry):
            return

        lock_timeout = self.lock_timeout

        if not self._acquire_lock(self.fine_payments_sync_origin, lock_timeout):
            LOGGER.error('cannot acquire lock')
            return

        with self._lock_releasing(self.fine_payments_sync_origin, lock_timeout):
            collect_kwargs = self._get_collect_kwargs(sync_entry)

            query = self._fine_collector.get_collectable_query(is_charged=True, is_charge_passed=False, **collect_kwargs)
            qs = AutocodeFine.objects.filter(query)
            limit = sync_entry.data.get('limit', None)

            for fine in qs:
                updated = self._fine_collector.check_fine_charged(fine)

                if limit is not None:
                    limit -= int(updated)
                    if limit <= 0:
                        break

                time.sleep(0.05)

    def collect_fine_payment_confirmations(self):
        """
        Collect all new fine payment notifications.
        """
        paid_fines_raw = self._autocode_client.get_new_paid_fines(limit=10**6)
        paid_fines = []

        for fine in paid_fines_raw:
            try:
                paid_fines.append(self._mark_fine_as_confirmed(fine))
            except Exception:
                LOGGER.exception('Exception while marking the fine as confirmed. Fine: `{}`'.format(fine))

        LOGGER.info('%d confirmations(s) had successfully been collected', len(paid_fines))

        return paid_fines

    def save_new_payment_confirmations(self, payment_confirmations):
        new_confirmation_ruling_numbers = [c.ruling_number for c in payment_confirmations]
        already_present_confirmatitons = (
            AutocodeFine.objects
            .filter(
                ruling_number__in=new_confirmation_ruling_numbers,
                payment_confirmation_received_at__isnull=False
            )
        )

        already_present_confirmation_ids = set()

        for confirmation in already_present_confirmatitons:
            already_present_confirmation_ids.add(confirmation.id)

        added_confirmations_number = 0
        for confirmation in payment_confirmations:
            if confirmation.id in already_present_confirmation_ids:
                continue
            confirmation.save()
            added_confirmations_number += 1
        LOGGER.info(
            '%d payment confirmation(s) had been saved successfully.',
            added_confirmations_number
        )

        return added_confirmations_number

    def remove_fines_from_feed(self, fine_objects):
        """
        Remove the set of fine and payment notifications from the feed.
        """
        autocode_ids = [o.autocode_id for o in fine_objects]
        self._autocode_client.remove_objects_from_feed(autocode_ids)

    def remove_confirmations_from_feed(self, confirmation_objects):
        autocode_ids = [o.autocode_payment_confirmation_id for o in confirmation_objects]
        self._autocode_client.remove_objects_from_feed(autocode_ids)

    def _mark_fine_as_confirmed(self, fine):
        fine_ruling_number = str(fine['rulingNumber'])
        fine_item = (
            AutocodeFine.objects
            .filter(
                ruling_number=fine_ruling_number
            )
            .first()
        )

        if fine_item is None:
            raise RuntimeError('Got payment confirmation for the fine which is not present in DB')

        fine_item.payment_confirmation_received_at = timezone.now()

        return fine_item

    def rebind_fines(self, since_ts):
        fine_ids = (
            AutocodeFine.objects
            .filter(Q(user_id__isnull=True) & Q(added_at_timestamp__gte=since_ts))
            .values_list('id', flat=True)
        )

        for fine_id in fine_ids:
            with transaction.atomic(savepoint=False):
                fine = AutocodeFine.objects.select_for_update().get(id=fine_id)
                self._try_rebind_fine(fine)
                fine.save()

        LOGGER.info('rebind have been finished successfully')

    def _try_rebind_fine(self, fine):
        assert isinstance(fine, AutocodeFine)
        if fine.violation_time is None:
            return

        needs_charge, session_id, skipped, user = self._bind_session(
            fine.ruling_number, fine.car, fine.violation_time, fine.is_camera_fixation
        )

        fine.session_id = session_id
        fine.skipped = skipped
        fine.user = user

        if fine.charged_at is None and fine.needs_charge != needs_charge:
            fine.needs_charge = needs_charge

        violation_latitude, violation_longitude = self._get_violation_location(fine.violation_time, session_id)
        fine.violation_latitude = violation_latitude
        fine.violation_longitude = violation_longitude

    def _create_new_fine_db_object(self, fine_dict):
        ruling_date = _parse_date(fine_dict['rulingDate'])

        fine_number = fine_dict['rulingNumber']
        violation_time = _parse_date_and_time(fine_dict['violationDateWithTime'])

        associated_car = self._get_associated_car(fine_dict)
        is_camera_fixation = self._check_is_camera_fixation(fine_dict)

        needs_charge, session_id, skipped, user = self._bind_session(
            fine_number, associated_car, violation_time, is_camera_fixation
        )
        violation_latitude, violation_longitude = self._get_violation_location(violation_time, session_id)

        article_koap = self._process_article_koap(fine_dict)

        discount = 50 if fine_dict['discountDate'] else 0  # percents
        discount_date = _parse_date(fine_dict['discountDate'])

        article_koap_title = article_koap.split(' - ')[0]
        if discount_date is None and article_koap_title in settings['force_discount']:
            discount = 50
            discount_date = ruling_date + datetime.timedelta(days=20)

        added_at = timezone.now()
        added_at_timestamp = int(added_at.timestamp())

        fine = AutocodeFine(
            ruling_number=fine_number,
            autocode_id=fine_dict['id'],
            article_koap=article_koap,
            violation_document_number=fine_dict['violationDocumentNumber'],
            odps_name=fine_dict['odpsName'],
            odps_code=fine_dict['odpsCode'],
            ruling_date=ruling_date,
            violation_time=violation_time,
            violation_place=fine_dict['violationPlace'],
            violation_latitude=violation_latitude,
            violation_longitude=violation_longitude,
            sum_to_pay=Decimal(fine_dict['sumToPay'] * (100 - discount) / 100),
            sum_to_pay_without_discount=Decimal(fine_dict['sumToPay']),
            discount_date=discount_date,
            has_photo=fine_dict['hasPhoto'],
            added_at_timestamp=added_at_timestamp,
            fine_information_received_at=added_at,
            car=associated_car,
            user=user,
            order=None,
            session_id=session_id,
            needs_charge=needs_charge,
            is_camera_fixation=is_camera_fixation,
            is_after_ride_start_during_order=False,
            skipped=skipped,
            source_type=AutocodeFine.SourceTypes.AUTOCODE.value,
        )

        return fine

    def _bind_session(self, fine_number, associated_car, violation_time, is_camera_fixation):
        violation_timestamp = _datetime_to_timestamp(violation_time)

        nearest_session_dict = self._get_associated_session(associated_car, violation_timestamp)

        # if not nearest_session_dict:
        #     if abs(fine_dict['sumToPay'] - 300000) < 1e-6:  # illegal parking fines
        #         LOGGER.info('try to bind fine {} using compiled history'.format(fine_number))
        #         nearest_session_dict = self._get_associated_session_custom(associated_car, violation_timestamp)

        if nearest_session_dict:
            skipped_limit = self.SKIPPED_LIMIT
            if violation_time <= pytz.timezone('Europe/Moscow').localize(
                    datetime.datetime(2018, 9, 7)  # bad telematics logs before
            ):
                skipped_limit = 0

            user = User.objects.filter(id=nearest_session_dict['user_id']).first()
            session_id = nearest_session_dict['session_id']
            skipped = nearest_session_dict['skipped']
            needs_charge = (is_camera_fixation and skipped <= skipped_limit)

            LOGGER.info('a session {} has been successfully matched for fine #{}'.format(session_id, fine_number))
        else:
            user = None
            session_id = None
            skipped = None
            needs_charge = False

            LOGGER.error('no order and session data to match for fine #{}'.format(fine_number))

        return needs_charge, session_id, skipped, user

    def _get_associated_car(self, fine_dict):
        if fine_dict['violationDocumentType'] == AutocodeFine.DocumentTypes.STS.value:
            sts_number = int(fine_dict['violationDocumentNumber'])
            associated_car = Car.objects.filter(registration_id=sts_number).first()
            if associated_car is None:
                associated_car = Car.objects.filter(
                    document_assignments__document__car_registry_document__registration_id=fine_dict[
                        'violationDocumentNumber']
                ).first()
        else:
            raise NotImplementedError('Currently we only fetch the data by STS.')

        if associated_car is None:
            raise RuntimeError('There is no car with STS number={}'.format(fine_dict['violationDocumentNumber']))

        return associated_car

    def _check_is_camera_fixation(self, fine_dict):
        ruling_number = fine_dict['rulingNumber']
        odps_name = fine_dict['odpsName']

        if 'ГИБДД' in odps_name:
            is_camera_fixation = (ruling_number[3:6] == '101')
        elif odps_name == 'МАДИ':
            is_camera_fixation = (ruling_number[8:11] == '101')
        elif odps_name == 'ГКУ "АМПП"':
            is_camera_fixation = False  # unpaid city parking
            # is_camera_fixation = (ruling_number[6:9] == '101')
        else:
            LOGGER.exception('fines: cant understand odpsName = %s', odps_name)
            is_camera_fixation = False

        return is_camera_fixation

    def _get_associated_session(self, associated_car, violation_timestamp):
        try:
            nearest_session_dict = self._saas_client.get_nearest_session(
                timestamp=violation_timestamp,
                car_id=associated_car.id,
                oauth_token=settings['saas_nearest_session_token'],
            )
        except Exception:
            LOGGER.exception('Failed to get nearest session from saas')
            nearest_session_dict = None

        return nearest_session_dict

    def _get_associated_session_custom(self, associated_car, violation_timestamp):
        try:
            nearest_session_dict = self._saas_client.get_nearest_session_custom(
                timestamp=violation_timestamp,
                car_id=associated_car.id,
                oauth_token=settings['saas_nearest_session_token'],
            )
        except Exception:
            LOGGER.exception('Failed to get nearest session from saas using compiled rides history')
            nearest_session_dict = None

        return nearest_session_dict

    def _get_violation_location(self, violation_time, session_id):
        violation_timestamp = _datetime_to_timestamp(violation_time)

        violation_latitude, violation_longitude = None, None

        last_point = None

        if session_id is not None:
            tracks = []

            try:
                tracks = self._saas_client.get_tracks(
                    order_id=session_id,
                    oauth_token=settings['saas_tracks_token']
                )
            except Exception:
                LOGGER.exception('failed to get saas tracks')

            for track in tracks:
                for tup in track:
                    # tup - tuple (timestamp, latitude, longitude)
                    # tracks are mixed: sometimes we cat get earlier track, that we don't need
                    if tup[0] <= violation_timestamp and (last_point is None or tup[0] >= last_point[0]):
                        last_point = tup
                    else:
                        break

        if last_point is not None:
            violation_latitude, violation_longitude = last_point[1:3]

        return violation_latitude, violation_longitude

    def _process_article_koap(self, fine_dict):
        article_koap = fine_dict['articleKoap']
        if re.match(
                r'Нарушение, предусмотренное.частью 4.настоящей статьи,'
                ' совершенное в городе федерального значения Москве или Санкт-Петербурге',
                article_koap
        ):
            article_koap = (
                '12.16.5 - Нарушение, предусмотренное частью 4 настоящей статьи, совершенное в городе '
                'федерального значения Москве или Санкт-Петербурге'
            )
        return article_koap
