import datetime
import logging
import pytz
from pprint import pformat

from django.db.models import Q

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.startrek_client import StartrekClient
from cars.fines.models.fine import AutocodeFine
from cars.fines.models.fine_report import AutocodeFineReport


LOGGER = logging.getLogger(__name__)


class FinesReporter(object):
    DEFAULT_USER_NAME = '**Нет информации**'
    USER_TICKET_TAG_NAME = 'violation_hardpdd_parking'
    DEFAULT_ST_QUEUE = 'illegalparking'

    DUPLICATE_REPORT_TEMPLATE = 'Found a fine #{} not reported automatically, but the issue describes it already'

    def __init__(self, startrek_client, saas_client):
        assert isinstance(startrek_client, StartrekClient)
        assert isinstance(saas_client, SaasDriveAdminClient)
        self._startrek_client = startrek_client
        self._saas_client = saas_client

    @classmethod
    def from_settings(cls):
        return cls(
            startrek_client=StartrekClient.from_settings(),
            saas_client=SaasDriveAdminClient.from_settings(),
        )

    def _format_title(self, fine):
        title_parts = [fine.ruling_number]

        if fine.user:
            title_parts.append(fine.user.get_full_name())
        else:
            title_parts.append(self.DEFAULT_USER_NAME)

        title_parts.append(fine.car.number)

        title = ' '.join(title_parts)
        return title

    def _format_description(self, fine):
        violation_time = None
        if fine.violation_time:
            violation_time = fine.violation_time.astimezone(pytz.timezone('Europe/Moscow')).strftime('%Y-%m-%d %H:%M:%S'),

        data = {
            'ruling_number': fine.ruling_number,
            'ruling_date': fine.ruling_date.strftime('%Y-%m-%d'),
            'car.number': fine.car.number,
            'vin': fine.car.vin,
            'article_koap': fine.article_koap,
            'violation_time': violation_time,
            'violation_place': fine.violation_place
        }

        if fine.order_id:
            data['order'] = 'https://carsharing.yandex-team.ru/#/orders/{}'.format(fine.order_id)
        elif fine.session_id:
            data['session'] = 'https://carsharing.yandex-team.ru/#/session/{}'.format(fine.session_id)

        return pformat(data)

    def get_lawn_parking_fines(self):
        already_reported = AutocodeFineReport.objects.values_list(
            'fine_id',
            flat=True
        )
        return AutocodeFine.objects.filter(
            Q(sum_to_pay_without_discount=300000)
            & ~Q(id__in=already_reported),
        ).order_by('fine_information_received_at')

    def report_fine_to_startrek(self, fine):
        try:
            issue_info = self._process_startrek_issue(fine)

            try:
                self._add_user_tags(issue_info, fine)
            except Exception:
                LOGGER.exception('error adding user tag processing fine {}'.format(fine.id))

            AutocodeFineReport.objects.create(fine=fine)
        except Exception:
            LOGGER.exception('error reporting fine {}'.format(fine.ruling_number))
        else:
            LOGGER.info('fine {} has been reported successfully'.format(fine.ruling_number))

    def _process_startrek_issue(self, fine):
        found_fines = self._startrek_client.search_issue(query=fine.ruling_number)

        existing_issue_keys = [
            f['key'] for f in found_fines
            if (
                f['key'].lower().startswith(self.DEFAULT_ST_QUEUE.lower()) and
                str(fine.ruling_number) in f['summary'].lower()
            )
        ]

        if existing_issue_keys:
            issue_info = self._notify_duplicated_issue(fine, existing_issue_keys)
        else:
            issue_info = self._make_issue(fine)

        return issue_info

    def _make_issue(self, fine):
        title = self._format_title(fine)

        description = self._format_description(fine)

        extra_info = {
            'rulingNumber': fine.ruling_number,
            'rulingDate': fine.ruling_date.strftime('%Y-%m-%d'),
        }

        deadline = fine.ruling_date + datetime.timedelta(days=5)
        if deadline.weekday() in (5, 6):
            deadline += datetime.timedelta(days=7 - deadline.weekday())

        extra_info['deadline'] = deadline.strftime('%Y-%m-%d')

        if fine.violation_time is not None:
            extra_info['evacuationDate'] = fine.violation_time.strftime('%Y-%m-%dT%H:%M:%S.000%z')

        if fine.violation_place:
            extra_info['violationPlace'] = fine.violation_place

        if fine.car_id is not None:
            extra_info['stateNumber'] = fine.car.number

        issue_info = self._startrek_client.create_issue(
            summary=title,
            description=description,
            queue=self.DEFAULT_ST_QUEUE,
            **extra_info
        )
        return issue_info

    def _notify_duplicated_issue(self, fine, existing_issue_keys):
        comment = self.DUPLICATE_REPORT_TEMPLATE.format(fine.ruling_number)

        for issue_key in existing_issue_keys:
            self._startrek_client.add_issue_comment(issue_key, text=comment)

        issue_info = self._startrek_client.get_issue(issue_key)
        return issue_info

    def _add_user_tags(self, issue_info, fine):
        user_id = fine.user_id

        if user_id is None:
            LOGGER.error('cannot add user tag as user is not defined: fine id - {}'.format(fine.id))
            return

        tag_name = self.USER_TICKET_TAG_NAME
        car_number = (fine.car.number or '') if fine.car_id is not None else ''
        session_id = fine.session_id or ''
        ticket_url = self._get_ticket_url(issue_info)

        if not self._check_tag_exists(user_id, tag_name, ticket_url):
            self._saas_client.add_user_problem_tag(
                user_id=user_id,
                tag=tag_name,
                car_number=car_number,
                session_id=session_id,
                st_links=ticket_url,
                re_raise=True,
            )
        else:
            LOGGER.info('user tag to add already exists: fine id - {}'.format(fine.id))

    def _get_ticket_url(self, ticket_info):
        return '{}/{}'.format(self._startrek_client.web_url, ticket_info['key'])

    def _check_tag_exists(self, user_id, tag_name, ticket_url):
        user_tags = self._saas_client.get_user_tags(user_id, re_raise=True)

        for tag_instance in user_tags:
            if tag_instance['tag'] == tag_name:
                links = tag_instance.get('links', [])
                for link in links:
                    if link['type'] == 'st' and link['uri'] == ticket_url:
                        return True

        return False
