import datetime
import logging
import re
import time

from decimal import Decimal

from django.db import transaction

from cars.core.solomon import SolomonHelper
from cars.core.util import datetime_helper
from cars.core.telephony import TelephonyHelperBase

import cars.settings

from cars.carsharing.models import Car
from cars.carsharing.models.tag import CarTag
from cars.fines.models import AutocodeFine
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin


LOGGER = logging.getLogger(__name__)


class YndxFinesApiHelper(TelephonyHelperBase):
    def __init__(self, api_token, request_timeout=15, retries=3):
        super().__init__(request_timeout, retries)
        self._setup_authorization_headers(api_token)

    def _setup_authorization_headers(self, api_token):
        self._session.headers.update({'Authorization': 'Token {}'.format(api_token)})

    @classmethod
    def from_settings(cls):
        api_token = cars.settings.FINES['yndx_fines']['token']
        return cls(api_token=api_token)

    def _perform_request(self, url, *, raise_for_status=False, **kwargs):
        response_data = super()._perform_request(url, raise_for_status=raise_for_status, **kwargs)

        status = response_data.get('status') if response_data else None

        if status == 'success':
            result = response_data['result']
        elif raise_for_status:
            raise Exception('error performing request: {}'.format(url))
        else:
            result = None

        return status, result

    def request_sts_check(self, sts_collection):
        url = 'https://money.yandex.ru/api/debts/v1/fines/createRequest'
        params = {'vehicleCertificates': sts_collection}
        _, result = self._perform_request(url, method='post', json=params, raise_for_status=False)
        request_id = result['requestId'] if result is not None else None
        return request_id

    def get_fine_check_data(self, request_id):
        """
        Example:
            {
                "status": "success",
                "result": {
                    "fines": [
                        {
                            "type": "VEHICLE_REG_CERTIFICATE",
                            "number": "9913829304",
                            "uin": "18810177190616045168",
                            "paymentLink": "https://money.yandex.ru/debts?uin=18810177190616045168&fineApiAgentId=13&ref=api",
                            "fine": {
                                "billDate": "2019-06-16T00:00:00+03:00",
                                "sum": 500.00,
                                "discountedSum": 250.00,
                                "discountDate": "2019-07-08T00:00:00+03:00",
                                "articleCode": "КоАП 12.9 ч.2, за превышение на 20-40 км/ч",
                                "location": "г. Москва"
                            }
                        }
                    ]
                }
            }
        """
        ready, fines_data = True, None

        url = 'https://money.yandex.ru/api/debts/v1/fines/getFines?requestId={}'.format(request_id)
        status, result = self._perform_request(url, raise_for_status=False, exc_info=False)

        if status == 'progress':
            ready = False
        else:
            fines_data = result['fines'] if result is not None else None

        return ready, fines_data


class YndxFinesStateWorker(object):
    TIME_STEP = 24 * 60 * 60
    MAX_CHUNK_SIZE = 10
    DEFAULT_REQUEST_INTERVAL = 30

    def __init__(self):
        self._default_since = datetime_helper.utc_localize(datetime.datetime(2019, 6, 20))
        self._sync_origin = SyncOrigin.YNDX_FINES

        self._api_helper = YndxFinesApiHelper.from_settings()
        self._solomon_helper = SolomonHelper('carsharing', 'yndx_fines_collecting')

    @classmethod
    def from_setting(cls):
        return cls()

    @property
    def sync_origin(self):
        return self._sync_origin.value

    def _get_sync_entry(self):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=self.sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=self.sync_origin, last_data_sync_time=self._default_since
                )

            if sync_entry.data is None:
                sync_entry.data = {}
                sync_entry.save()

        return sync_entry

    def _get_sync_entry_for_update(self):
        sync_entry = CallStatSyncStatus.objects.select_for_update().get(origin=self.sync_origin)
        return sync_entry

    def check_is_active(self, sync_entry=None):
        if sync_entry is None:
            sync_entry = self._get_sync_entry()

        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now

        is_active = started and not finished
        return is_active

    def prepare_fines_to_process(self):
        sync_entry = self._get_sync_entry()

        if not self.check_is_active(sync_entry):
            return

        self._report_state(sync_entry)

        now = datetime_helper.timestamp_now()

        if sync_entry.data.get('next_sync_timestamp', 0) >= now:
            return

        fake_car_ids = list(CarTag.objects.filter(tag='fake_car').values_list('object_id', flat=True))

        sts_collection_iter = (
            Car.objects
            .filter(registration_id__isnull=False)
            .exclude(id__in=fake_car_ids)
            .values_list('registration_id', flat=True)
        )

        sts_collection = []

        for sts in sts_collection_iter:
            if sts > 5000000000:
                sts_collection.append(sts)
            else:
                LOGGER.warning('there is a car with incorrect registration id {}'.format(sts))

        with transaction.atomic(savepoint=False):
            sync_entry = self._get_sync_entry_for_update()
            sync_entry.data['sts_to_check'] = sts_collection
            sync_entry.data['next_sync_timestamp'] = now + self.TIME_STEP
            sync_entry.save()

    def _report_state(self, sync_entry):
        sts_to_check = sync_entry.data.get('sts_to_check', [])
        self._solomon_helper.report_value('sts_to_check', len(sts_to_check))

        requests_to_get = sync_entry.data.get('requests_to_get', [])
        self._solomon_helper.report_value('pending_requests', len(requests_to_get))

    def process_fines(self):
        sync_entry = self._get_sync_entry()

        if not self.check_is_active(sync_entry):
            return

        sts_to_check = sync_entry.data.get('sts_to_check', [])

        chunk_size = self._calculate_chunk_size(sync_entry)
        request_interval = self._calculate_request_interval(sync_entry)

        for offset in range(0, len(sts_to_check), chunk_size):
            chunk = sts_to_check[offset:offset + chunk_size]

            request_id = self._api_helper.request_sts_check(chunk)

            if request_id is not None:
                with transaction.atomic():
                    sync_entry = self._get_sync_entry_for_update()
                    sync_entry.data['sts_to_check'] = [x for x in sync_entry.data['sts_to_check'] if x not in chunk]
                    sync_entry.data.setdefault('requests_to_get', [])
                    sync_entry.data['requests_to_get'].append([request_id, chunk])
                    sync_entry.save()

            time.sleep(request_interval)

        LOGGER.info('all requests to check sts have been sent')

    def _calculate_chunk_size(self, sync_entry):
        next_sync_timestamp = sync_entry.data.get('next_sync_timestamp', 0.0)
        chunk_size = sync_entry.data.get('max_chunk_size', self.MAX_CHUNK_SIZE)
        chunk_decreasing_ratio = sync_entry.data.get('chunk_decreasing_ratio', 1.0)

        time_remained_ratio = (next_sync_timestamp - datetime_helper.timestamp_now()) / self.TIME_STEP
        time_remained_ratio /= chunk_decreasing_ratio

        if 0 < time_remained_ratio < 1:
            super_power_of_two = 1 << (int.bit_length(int(1 / time_remained_ratio)) - 1)
            chunk_size = max(1, (chunk_size // super_power_of_two))

        return chunk_size

    def _calculate_request_interval(self, sync_entry):
        request_interval = sync_entry.data.get('request_interval', self.DEFAULT_REQUEST_INTERVAL)
        return request_interval

    def process_result(self):
        sync_entry = self._get_sync_entry()

        if not self.check_is_active(sync_entry):
            return

        results_to_process = sync_entry.data.get('requests_to_get', [])

        for request_id, sts_collection in results_to_process:
            ready, fines_data = self._api_helper.get_fine_check_data(request_id)

            if not ready:
                continue

            LOGGER.info('processing request {} results'.format(request_id))

            if fines_data is not None:
                LOGGER.info(
                    'data has been successfully obtained for sts: {} (request id - {})'
                    .format(sts_collection, request_id)
                )

                existing_ruling_numbers = set(
                    AutocodeFine.objects
                    .filter(ruling_number__in=[fine_data['uin'] for fine_data in fines_data])
                    .values_list('ruling_number', flat=True)
                )

                for fine_data in fines_data:
                    if fine_data['uin'] not in existing_ruling_numbers:
                        self._process_fines_data(fine_data)
            else:
                LOGGER.info(
                    'failed to obtain data for sts: {} (request id - {})'.format(sts_collection, request_id)
                )

            with transaction.atomic():
                sync_entry = self._get_sync_entry_for_update()

                if fines_data is None:
                    sync_entry.data['sts_to_check'].extend(sts_collection)

                sync_entry.data['requests_to_get'] = [
                    x for x in sync_entry.data['requests_to_get'] if x[0] != request_id
                ]
                sync_entry.save()

    def _process_fines_data(self, fine_data):
        assert fine_data['type'] == 'VEHICLE_REG_CERTIFICATE'

        violation_document_number = fine_data['number']
        violation_document_type = AutocodeFine.DocumentTypes.STS.value

        ruling_number = fine_data['uin']
        ruling_date = self._parse_date(fine_data['fine']['billDate'])
        sum_to_pay_without_discount = Decimal(fine_data['fine']['sum'])

        discount_date = self._parse_date(fine_data['fine'].get('discountDate', None))
        sum_to_pay = Decimal(fine_data['fine'].get('discountedSum', sum_to_pay_without_discount))

        article_koap = fine_data['fine'].get('articleCode', '')
        violation_place = fine_data['fine'].get('location', '')

        car = Car.objects.get(registration_id=violation_document_number)
        fine_information_received_at = datetime_helper.utc_now()

        has_photo = False
        needs_charge = False
        is_after_ride_start_during_order = False
        is_camera_fixation = False

        source_type = AutocodeFine.SourceTypes.YNDX_FINES.value

        AutocodeFine.objects.create(
            violation_document_number=violation_document_number,
            violation_document_type=violation_document_type,
            ruling_number=ruling_number,
            ruling_date=ruling_date,
            sum_to_pay_without_discount=sum_to_pay_without_discount,
            discount_date=discount_date,
            sum_to_pay=sum_to_pay,
            article_koap=article_koap,
            violation_place=violation_place,
            car=car,
            fine_information_received_at=fine_information_received_at,
            added_at_timestamp=int(fine_information_received_at.timestamp()),
            has_photo=has_photo,
            needs_charge=needs_charge,
            is_after_ride_start_during_order=is_after_ride_start_during_order,
            is_camera_fixation=is_camera_fixation,
            source_type=source_type,
        )

    def _parse_date(self, value):
        if value is None:
            return None

        value = re.sub('([+-]\d{2}):(\d{2})', r'\1\2', value)
        parsed_timestamp = datetime.datetime.strptime(value, '%Y-%m-%dT%H:%M:%S%z')

        parsed_date = parsed_timestamp.date()
        return parsed_date
