import logging

from django.db.models import Max
from django.utils import timezone

from cars.core.autocode import AutoCodeFinesGetter
from cars.core.daemons import CarsharingDaemon, CarsharingWorker

from ..core.fines_manager import FinesManager
from ..core.yndx_fines_manager import YndxFinesStateWorker
from ..models.fine import AutocodeFine


LOGGER = logging.getLogger(__name__)


class FinesCollectorDaemon(CarsharingDaemon):
    def get_distributed_lock_relative_path(self):
        return 'carsharing/locks/fines_collector.lock'

    def get_solomon_service(self):
        return 'carsharing'

    get_solomon_sensor_prefix = None
    _do_tick = None

    def init_workers(self):
        self.register_worker('autocode_cars_subscription', AutocodeFineCarSubscribeWorker)
        # self.register_worker('autocode_fine_payment_processing', AutocodeFinePaymentProcessorWorker)

        # deprecated processors
        # self.register_worker('autocode_collecting', AutocodeFineCollectingWorker)
        # self.register_worker('autocode_missing_photos_collecting', AutocodeFinePhotoCollectingWorker)

        # self.register_worker('yndx_fines_collecting_prepare', YndxFineCollectingPrepareWorker)
        # self.register_worker('yndx_fines_collecting_request', YndxFineCollectingRequestWorker)
        # self.register_worker('yndx_fines_collecting_process', YndxFineCollectingProcessWorker)


# the processor is deprecated and disabled now
class FinesProcessorDaemon(CarsharingDaemon):
    tick_interval = '* 7-17 * * *'  # every minute since 10 till 21 (utc +0300)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = FinesManager.from_settings()

    def get_distributed_lock_relative_path(self):
        return 'carsharing/locks/fines_processor.lock'

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fines_processor'

    def get_solomon_service(self):
        return 'carsharing'

    def _do_tick(self):
        self._fines_manager.process_existing_fines()


class AutocodeFineCarSubscribeWorker(CarsharingWorker):
    tick_interval = '30 8-18 * * *'  # every hour from 11 to 21 utc+3 (8 to 18 utc)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = FinesManager.from_settings()
        self._autocode_client = AutoCodeFinesGetter.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fine_cars_subscriber'

    def _do_tick(self):
        try:
            self._fines_manager.subscribe_all_cars()
        except Exception as exc:
            LOGGER.exception('failed to subscribe cars')
            raise


class AutocodeFineCollectingWorker(CarsharingWorker):
    tick_interval = '30 8-18 * * *'  # every hour from 11 to 21 utc+3 (8 to 18 utc)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = FinesManager.from_settings()
        self._autocode_client = AutoCodeFinesGetter.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fines_collector'

    def _do_tick(self):
        exc = None

        fines, exc = self._collect_autocode_fines(exc)

        if fines:
            self._fines_manager.remove_fines_from_feed(fines)

        payment_confirmations, exc = self._collect_autocode_payment_confirmations(exc)

        if payment_confirmations:
            self._fines_manager.remove_confirmations_from_feed(payment_confirmations)

        exc = self._attach_autocode_missing_photos(exc)
        exc = self._update_autocode_feed(exc)

        if exc:
            raise exc

    def _collect_autocode_fines(self, exc):
        fines = None

        try:
            today = timezone.now().date()
            last_fine_created_date = (
                AutocodeFine.objects.filter(source_type=AutocodeFine.SourceTypes.AUTOCODE.value)
                .aggregate(Max('fine_information_received_at'))
                ['fine_information_received_at__max'].date()
            )

            if last_fine_created_date == today:
                LOGGER.info('there are fresh autocode fines in base, no need to disturb autocode')
            else:
                LOGGER.info('getting new fines...')
                raw_fines = self._fines_manager.collect_new_fines()
                fines, added_fines_count = self._fines_manager.save_new_fines(raw_fines)

        except Exception as e:
            fines = None  # ensure that fines will be obtained again
            LOGGER.exception('failed to get new fines')
            exc = e

        return fines, exc

    def _collect_autocode_payment_confirmations(self, exc):
        payment_confirmations = None

        try:
            today = timezone.now().date()
            last_confirmation_date = (
                AutocodeFine.objects.filter(source_type=AutocodeFine.SourceTypes.AUTOCODE.value)
                .aggregate(Max('payment_confirmation_received_at'))
                ['payment_confirmation_received_at__max'].date()
            )

            if last_confirmation_date == today:
                LOGGER.info('there are fresh autocode confirmations in base, not asking for new')
            else:
                LOGGER.info('NOT getting new payment confirmations')
                # LOGGER.info('getting new payment confirmations')
                # payment_confirmations = self._fines_manager.collect_fine_payment_confirmations()
                # self._fines_manager.save_new_payment_confirmations(payment_confirmations)

        except Exception as e:
            LOGGER.exception('failed to get new payment confirmations')
            exc = e

        return payment_confirmations, exc

    def _attach_autocode_missing_photos(self, exc):
        try:
            LOGGER.info('attaching missing photos')
            self._fines_manager.attach_missing_photos()
        except Exception as e:
            LOGGER.exception('failed to attach new missing photos')
            exc = e

        return exc

    def _update_autocode_feed(self, exc):
        try:
            self._fines_manager.subscribe_all_cars()
        except Exception as e:
            LOGGER.exception('failed to subscribe cars')
            exc = e

        return exc


class AutocodeFinePhotoCollectingWorker(CarsharingWorker):
    tick_interval = '*/5 * * * *'  # every 5 minutes

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = FinesManager.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fine_photo_collector'

    def _do_tick(self):
        try:
            LOGGER.info('attaching missing photos')
            self._fines_manager.attach_missing_photos()
        except Exception as e:
            LOGGER.exception('failed to attach new missing photos')
            raise


class AutocodeFinePaymentProcessorWorker(CarsharingWorker):
    tick_interval = '*/10 * * * *'  # every 10 minutes

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = FinesManager.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fine_payment_processor'

    def _do_tick(self):
        try:
            LOGGER.info('processing fine payments')
            self._fines_manager.process_fine_payments()
        except Exception as e:
            LOGGER.exception('failed to process fine payments')
            raise


class YndxFineCollectingPrepareWorker(CarsharingWorker):
    tick_interval = '* * * * *'  # every minute

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = YndxFinesStateWorker.from_setting()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.yndx_fines_collecting.prepare'

    def _do_tick(self):
        self._fines_manager.prepare_fines_to_process()


class YndxFineCollectingRequestWorker(CarsharingWorker):
    tick_interval = '* * * * * */30'  # every half a minute

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = YndxFinesStateWorker.from_setting()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.yndx_fines_collecting.request'

    def _do_tick(self):
        self._fines_manager.process_fines()


class YndxFineCollectingProcessWorker(CarsharingWorker):
    tick_interval = '* * * * * */30'  # every half a minute

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_manager = YndxFinesStateWorker.from_setting()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.yndx_fines_collecting.process'

    def _do_tick(self):
        self._fines_manager.process_result()
