import logging

import croniter

from cars.core.daemons import CarsharingDaemon, CarsharingWorker
from cars.core.util import datetime_helper
from ..core.fines_reporter import FinesReporter
from ..core.fine_exporter import FinesExportHelper


LOGGER = logging.getLogger(__name__)


class FinesReporterDaemon(CarsharingDaemon):
    def get_distributed_lock_relative_path(self):
        return 'carsharing/locks/fines_reporter.lock'

    def get_solomon_service(self):
        return 'carsharing'

    get_solomon_sensor_prefix = None
    _do_tick = None

    def init_workers(self):
        self.register_worker('fines_reporting', FinesReportingWorker)

        self.register_worker('fines_export', FinesExportingWorker)


class FinesReportingWorker(CarsharingWorker):
    tick_interval = '0 * * * *'  # every hour

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fines_reporter = FinesReporter.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fines_reporter'

    def _do_tick(self):
        fines = self._fines_reporter.get_lawn_parking_fines()

        for f in fines:
            self._fines_reporter.report_fine_to_startrek(f)


class FinesExportingWorker(CarsharingWorker):
    tick_interval = '* * * * *'  # every minute
    optimization_tick_interval = '0 */12 * * *'  # every 12 hours
    cars_update_tick_interval = '0 */3 * * *'  # every 3 hours

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._export_helper = FinesExportHelper.from_settings()

        self._optimization_cron = croniter.croniter(self.optimization_tick_interval)
        self._next_optimization_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._optimization_cron)
        )

        self._cars_update_cron = croniter.croniter(self.cars_update_tick_interval)
        self._next_cars_update_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._cars_update_cron)
        )

    def get_solomon_sensor_prefix(self):
        return 'carsharing.fines_exporter'

    def _do_tick(self):
        if datetime_helper.utc_now() >= self._next_cars_update_time:
            self._export_helper.init_car_sources()
            self._next_cars_update_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._cars_update_cron)
            )

        self._export_helper.export()

        if datetime_helper.utc_now() >= self._next_optimization_time:
            self._export_helper.optimize()
            self._next_optimization_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._optimization_cron)
            )
