import collections
import datetime
import logging

from django.db.models import Max, Avg
from django.utils import timezone

from cars.core.daemons import CarsharingDaemon
from ..models.fine import AutocodeFine
from ..core.fines_manager import FinesManager


LOGGER = logging.getLogger(__name__)


Interval = collections.namedtuple('Interval', ['label', 'timedelta'])


# the processor is deprecated and disabled now
class FinesMonitoringDaemon(CarsharingDaemon):

    tick_interval = '*/30 * * * *'  # every 30 minutes

    def get_distributed_lock_relative_path(self):
        return 'fines/locks/monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'fines.monitoring'

    def get_solomon_service(self):
        return 'fines'

    def _do_tick(self):
        exc = None
        for monitor_name, monitor_callback in [
                ('new fines', self._monitor_new_fines),
                ('unpaid fines', self._monitor_unpaid_fines),
                ('fines max value', self._monitor_fines_max_value),
                ('fines without email notification',
                 self._monitor_fines_no_email_notification),
                ('fines without sms notification',
                 self._monitor_fines_no_sms_notification),
                ('avg skipped value', self._monitor_fines_avg_skipped),
                ('max skipped value', self._monitor_fines_max_skipped),
                ('skipped bellow limit count',
                 self._monitor_fines_skipped_bellow_limit_count),
                ('skipped above limit count',
                 self._monitor_fines_skipped_above_limit_count),

        ]:
            try:
                monitor_callback()
            except Exception as e:
                LOGGER.exception('failed to monitor %s', monitor_name)
                exc = e
        if exc:
            raise exc

    def _monitor_new_fines(self):
        '''Fines received for day.'''
        new_fines = (
            AutocodeFine.objects
            .filter(fine_information_received_at__gte=timezone.now()
                    - datetime.timedelta(days=1))
            .count()
        )
        self.solomon.set_value(
            'orders.fines.new_fines',
            new_fines
        )

    def _monitor_unpaid_fines(self):
        '''Unpaid fines.'''
        unpaid_fines = (
            AutocodeFine.objects
            .filter(charged_at__isnull=True)
            .count()
        )
        self.solomon.set_value(
            'orders.fines.unpaid_fines',
            unpaid_fines
        )

    def _monitor_fines_no_email_notification(self):
        '''Payed fines without email notification.'''
        no_email_notification_fines = (
            AutocodeFine.objects
            .filter(charged_at__isnull=False,
                    charge_email_sent_at__isnull=True)
            .count()
        )
        self.solomon.set_value(
            'orders.fines.no_email_notification_fines',
            no_email_notification_fines
        )

    def _monitor_fines_no_sms_notification(self):
        '''Payed fines without sms notification.'''
        no_sms_notification_fines = (
            AutocodeFine.objects
            .filter(charged_at__isnull=False,
                    charge_sms_sent_at__isnull=True)
            .count()
        )
        self.solomon.set_value(
            'orders.fines.no_sms_notification_fines',
            no_sms_notification_fines
        )

    def _monitor_fines_max_value(self):
        '''Max fine amount.'''
        max_sum_to_pay = AutocodeFine.objects.aggregate(
            Max('sum_to_pay')
        )['sum_to_pay__max'] or 0
        self.solomon.set_value(
            'orders.fines.max_sum_to_pay',
            int(max_sum_to_pay)
        )

    def _monitor_fines_avg_skipped(self):
        avg_skipped = AutocodeFine.objects.filter(
            skipped__isnull=False
        ).aggregate(
            Avg('skipped')
        )['skipped__avg'] or 0
        self.solomon.set_value(
            'orders.fines.avg_skipped',
            avg_skipped,
        )

    def _monitor_fines_max_skipped(self):
        max_skipped = AutocodeFine.objects.filter(
            skipped__isnull=False
        ).aggregate(
            Max('skipped')
        )['skipped__max'] or 0
        self.solomon.set_value(
            'orders.fines.max_skipped',
            max_skipped,
        )

    def _monitor_fines_skipped_bellow_limit_count(self):
        skipped_bellow_limit_count = AutocodeFine.objects.filter(
            skipped__isnull=False,
            skipped__lte=FinesManager.SKIPPED_LIMIT,
        ).count()
        self.solomon.set_value(
            'orders.fines.skipped_bellow_limit_count',
            skipped_bellow_limit_count,
        )

    def _monitor_fines_skipped_above_limit_count(self):
        skipped_above_limit_count = AutocodeFine.objects.filter(
            skipped__isnull=False,
            skipped__gt=FinesManager.SKIPPED_LIMIT,
        ).count()
        self.solomon.set_value(
            'orders.fines.skipped_above_limit_count',
            skipped_above_limit_count,
        )
