import datetime
import pytz
from string import digits
from uuid import uuid4

from factory import LazyAttribute, LazyFunction, SelfAttribute, SubFactory
from factory.django import DjangoModelFactory
from factory.fuzzy import FuzzyText, FuzzyDecimal, FuzzyDecimal, FuzzyDate, FuzzyDateTime

from cars.carsharing.factories.car import CarFactory
from cars.users.factories.user import UserFactory
from cars.orders.factories.order import OrderFactory

from ..models.fine import AutocodeFine


class FineFactory(DjangoModelFactory):

    class Meta:
        model = AutocodeFine

    id = LazyFunction(uuid4)
    ruling_number = FuzzyText(chars=digits, length=10)
    autocode_id = FuzzyDecimal(10**6)
    autocode_payment_confirmation_id = None
    violation_time = FuzzyDateTime(
        datetime.datetime(2018, 2, 20, tzinfo=pytz.timezone('Europe/Moscow')),
        datetime.datetime.now(datetime.timezone.utc) - datetime.timedelta(hours=2))
    ruling_date = FuzzyDate(datetime.date(2018, 2, 20))
    discount_date = None
    article_koap = FuzzyText(length=20, prefix='СТАТЬЯ КОАП ')
    violation_place = FuzzyText(length=20, prefix='МЕСТО НАРУШЕНИЯ ')
    sum_to_pay_without_discount = FuzzyDecimal(100, 5000, precision=2)
    sum_to_pay = LazyAttribute(lambda obj: round(obj.sum_to_pay_without_discount / 2, 2))
    odps_code = FuzzyText(chars=digits, length=5)
    odps_name = FuzzyText(length=20, prefix='ОТДЕЛЕНИЕ ДПС ')
    violation_document_number = FuzzyText(chars=digits, length=10, prefix='777')
    violation_document_type = 'Sts'
    has_photo = False

    fine_information_received_at = (datetime.datetime.now(datetime.timezone.utc)
                                    - datetime.timedelta(hours=1))
    payment_confirmation_received_at = None

    needs_charge = True
    charged_at = None
    charge_email_sent_at = None
    charge_sms_sent_at = None

    car = SubFactory(CarFactory)
    user = SubFactory(UserFactory)
    order = SubFactory(
        OrderFactory,
        user=SelfAttribute('..user'),
    )
