import collections
import enum
import json
import uuid

from django.db import models

from cars.users.models import User
from cars.orders.models import Order
from cars.carsharing.models.car import Car


class AutocodeFine(models.Model):

    class SourceTypes(enum.Enum):
        AUTOCODE = 'autocode'
        YNDX_FINES = 'yndx_fines'

    class DocumentTypes(enum.Enum):
        DRIVER_LICENSE = 'DriverLicense'
        STS = 'Sts'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    serial_id = models.IntegerField(null=False)

    meta_info = models.TextField(null=True)

    # Information from source
    source_type = models.TextField(null=True)  # one of but not limited to SourceTypes
    ruling_number = models.CharField(max_length=64, unique=True)

    added_at_timestamp = models.BigIntegerField(null=False)

    autocode_id = models.BigIntegerField(unique=True, null=True)
    autocode_payment_confirmation_id = models.BigIntegerField(unique=True, null=True)
    violation_time = models.DateTimeField(null=True)
    ruling_date = models.DateField()
    discount_date = models.DateField(null=True)
    article_koap = models.TextField()
    violation_place = models.TextField()
    violation_latitude = models.FloatField(null=True)
    violation_longitude = models.FloatField(null=True)
    sum_to_pay = models.DecimalField(decimal_places=2, max_digits=9)
    sum_to_pay_without_discount = models.DecimalField(decimal_places=2, max_digits=9)
    odps_code = models.CharField(max_length=16, null=True)
    odps_name = models.TextField(null=True)
    violation_document_number = models.CharField(max_length=16)
    violation_document_type = models.CharField(
        max_length=13,
        choices=[(x.value, x.name) for x in DocumentTypes],
    )
    has_photo = models.BooleanField()

    # Service fields
    fine_information_received_at = models.DateTimeField()
    payment_confirmation_received_at = models.DateTimeField(null=True)

    needs_charge = models.BooleanField(default=False)  # some fines are payed by users, not by us
    charged_at = models.DateTimeField(null=True)  # money charge started
    charge_passed_at = models.DateTimeField(null=True)  # all the money belong to us
    charge_email_sent_at = models.DateTimeField(null=True)
    charge_sms_sent_at = models.DateTimeField(null=True)
    charge_push_sent_at = models.DateTimeField(null=True)

    is_camera_fixation = models.BooleanField(default=False)  # others are fixated manually and written to user
    # violation happened during the users order (not left from previous user)
    is_after_ride_start_during_order = models.BooleanField(default=False)

    # debug info
    skipped = models.IntegerField(null=True)  # returned by saas handler: skipped sessions before found appropriate

    session_id = models.CharField(max_length=36, null=True, default=None)
    car = models.ForeignKey(
        Car,
        related_name='autocode_fines',
        on_delete=models.CASCADE,
    )

    user = models.ForeignKey(
        User,
        related_name='autocode_fines',
        on_delete=models.CASCADE,
        null=True,
    )

    order = models.ForeignKey(
        Order,
        related_name='autocode_fines',
        on_delete=models.CASCADE,
        null=True,
    )

    class Meta:
        db_table = 'autocode_fine'

    def generate_duplicate_name(self):
        random_suffix = uuid.uuid4().hex[:20]
        generated_ruling_number = '_'.join([self.ruling_number, 'dup', random_suffix])
        return generated_ruling_number

    @property
    def parsed_meta_info(self):
        if self.meta_info is None:
            return None

        try:
            parsed_meta_info = json.loads(self.meta_info)
        except Exception:
            parsed_meta_info = None

        return parsed_meta_info

    @parsed_meta_info.setter
    def parsed_meta_info(self, value):
        if value is None:
            self.meta_info = None
        else:
            assert isinstance(value, collections.Mapping)
            self.meta_info = json.dumps(value)

    def update_meta_info(self, **kwargs):
        meta_info = self.parsed_meta_info or {}
        meta_info.update(**kwargs)
        self.parsed_meta_info = meta_info


class AutocodeFinePhoto(models.Model):

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    fine = models.ForeignKey(
        AutocodeFine,
        related_name='photos',
        on_delete=models.CASCADE,
        null=False,
    )

    url = models.CharField(max_length=256, null=False)

    class Meta:
        db_table = 'autocode_fine_photo'
