import collections
import operator


def merge_two_lists(left, right, *, key=None, reversed_source=False, reverse_result=False):
    key = key or (lambda x: x)

    result = []
    _compare = operator.gt if not reversed_source else operator.lt

    while left and right:
        c = left if _compare(key(left[-1]), key(right[-1])) else right
        result.append(c.pop())

    result.extend(reversed(left or right))

    result = list(reversed(result)) if not reverse_result else result

    return result


class MetaInfoVerifier(object):
    def verify(self, meta_info):
        raise NotImplementedError()


class NodeMetaInfoVerifier(MetaInfoVerifier):
    def verify(self, meta_info):
        verified_keys = set()

        labels = meta_info.get('labels', None)

        if not isinstance(labels, collections.Mapping):
            raise Exception('node labels must be a mapping')
        else:
            verified_keys.add('labels')

        verified_keys.update([
            'call_direction', 'enabled', 'order', 'permissions', 'origins', 'keywords',
        ])

        remaining_keys = [k for k in meta_info if k not in verified_keys]

        if remaining_keys:
            raise Exception('meta info contains extra keys: {}'.format(remaining_keys))


class EdgeMetaInfoVerifier(MetaInfoVerifier):
    def verify(self, meta_info):
        pass


class Tree(object):
    node_meta_info_verifier = NodeMetaInfoVerifier()
    edge_meta_info_verifier = EdgeMetaInfoVerifier()

    def __init__(self):
        self._tree = {}  # id: node in the tree mapping

    @staticmethod
    def make_node(id_, meta_info):
        return {
            'id': str(id_),
            'meta_info': meta_info,
            'parent_ids': [],
            'children': [],
        }

    @classmethod
    def check_node_meta_info(cls, meta_info):
        cls.node_meta_info_verifier.verify(meta_info)

    @classmethod
    def check_edge_meta_info(cls, meta_info):
        cls.edge_meta_info_verifier.verify(meta_info)

    def get_node(self, node_id, default=None):
        return self._tree.get(str(node_id), default)

    def exists(self, node_id):
        return str(node_id) in self._tree

    def is_root_node(self, node_id):
        return not self._tree[str(node_id)]['parent_ids']

    def iter_root_nodes(self):
        for node in self._tree.values():
            if not node['parent_ids']:
                yield node

    def iter_all_nodes(self, parent_id=None):
        for node in self._tree.values():
            if parent_id is None or parent_id in node['parent_ids']:
                yield node

    def filter_nodes(self, node_filter, node_iter=None):
        assert isinstance(node_filter, collections.Callable)

        if node_iter is None:
            node_iter = self.iter_all_nodes()

        for node in node_iter:
            if node_filter(node):
                yield node

    def iter_parents(self, node_id):
        for parent_id in self._tree[str(node_id)]['parent_ids']:
            yield self._tree[parent_id]

    def iter_children(self, node_id):
        yield from self._tree[str(node_id)]['children']

    def find_node_path(self, node_id):
        path = [self.get_node(node_id)]

        parents = path[-1]['parent_ids']

        # find the topmost path, multiple paths are not processed
        while parents:
            parent = self.get_node(parents[0])
            path.append(parent)
            parents = parent['parent_ids']

        path.reverse()
        return path

    def add_edge(self, parent_id, child_id):
        parent_id, child_id = str(parent_id), str(child_id)
        parent, child = self._tree[parent_id], self._tree[child_id]

        if child in parent['children']:
            raise Exception('edge between {} and {} already exists'.format(parent_id, child_id))

        parent['children'].append(child)
        child['parent_ids'].append(parent_id)

    def remove_edge(self, parent_id, child_id):
        parent_id, child_id = str(parent_id), str(child_id)
        parent, child = self._tree[parent_id], self._tree[child_id]

        if child not in parent['children']:
            raise Exception('edge between {} and {} does not exist'.format(parent_id, child_id))

        parent['children'].remove(child)
        child['parent_ids'].remove(parent_id)

    def add_node(self, node):
        self._tree[node['id']] = node

    def modify_node(self, updated_node):
        node = self._tree[updated_node['id']]
        node['meta_info'] = updated_node['meta_info']

    def remove_node(self, node_id):
        self._tree.pop(str(node_id))

    def is_acyclic(self, start_node_id=None):
        if start_node_id is not None:
            current_node_ids = [str(start_node_id)]
        else:
            current_node_ids = [x['id'] for x in self.iter_root_nodes()]

        # None - not visited, False - not finished, True - finished
        node_marks = {node_id: None for node_id in self._tree}

        while current_node_ids:
            current_node_id = current_node_ids[-1]

            if node_marks[current_node_id] is None:
                node_marks[current_node_id] = False

                children = (x['id'] for x in self._tree[current_node_id]['children'])

                for child in children:
                    if node_marks[child] is None:
                        current_node_ids.append(child)
                    elif node_marks[child] is False:
                        return False
                    else:
                        pass  # already visited

            elif node_marks[current_node_id] is False:
                node_marks[current_node_id] = True
                current_node_ids.pop()

            else:
                raise Exception('undefined behavior finding a cycle in a tree')

        return True
