import logging

from django.db import transaction

from cars.knowledge_base.core.category_tree.common_helper import Tree
from cars.knowledge_base.models import (
    RequestCategoryTreeEdge,
    RequestCategoryTreeEdgeHistory,
    RequestCategoryTreeNode,
    RequestCategoryTreeNodeHistory,
)
from cars.knowledge_base.models.categories import Action


LOGGER = logging.getLogger(__name__)


class CategoryTreeDBHelper(object):
    def __init__(self, tree):
        assert isinstance(tree, Tree)
        self._tree = tree

    @property
    def tree(self):
        return self._tree

    @tree.setter
    def tree(self, value):
        assert isinstance(value, Tree)
        self._tree = value

    def make_node(self, meta_info, performed_at, performed_by):
        node = RequestCategoryTreeNode.objects.create(
            meta_info=meta_info
        )

        self.tree.add_node(self.tree.make_node(node.category_id, {}))

        RequestCategoryTreeNodeHistory.objects.create(
            history_action=Action.ADD.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            category_id=node.category_id,
            meta_info=node.meta_info,
        )

        return node

    def alter_node(self, node, meta_info, performed_at, performed_by):
        RequestCategoryTreeNodeHistory.objects.create(
            history_action=Action.MODIFY.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            category_id=node.category_id,
            meta_info=node.meta_info,
        )

        node.meta_info = meta_info
        node.save()

        return node

    def destroy_node(self, node, performed_at, performed_by):
        RequestCategoryTreeNodeHistory.objects.create(
            history_action=Action.REMOVE.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            category_id=node.category_id,
            meta_info=node.meta_info,
        )

        self.tree.remove_node(node.category_id)

        node.delete()

    def make_edge(self, child_node, parent_node, performed_at, performed_by):
        self.tree.add_edge(parent_node.category_id, child_node.category_id)

        if not self.tree.is_acyclic(parent_node.category_id):
            raise Exception('edge between {} and {} induces a cycle in the graph'
                            .format(parent_node.category_id, child_node.category_id))

        edge = RequestCategoryTreeEdge.objects.create(
            parent=parent_node,
            child=child_node,
        )

        RequestCategoryTreeEdgeHistory.objects.create(
            history_action=Action.ADD.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            edge_id=edge.edge_id,
            parent_id=parent_node.category_id,
            child_id=child_node.category_id,
            meta_info=edge.meta_info,
        )

        return edge

    def alter_edge(self, edge, meta_info, performed_at, performed_by):
        RequestCategoryTreeEdgeHistory.objects.create(
            history_action=Action.MODIFY.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            edge_id=edge.edge_id,
            parent_id=edge.parent.category_id,
            child_id=edge.child.category_id,
            meta_info=edge.meta_info,
        )

        edge.meta_info = meta_info
        edge.save()

        return edge

    def destroy_edge(self, child_node, parent_node, performed_at, performed_by, *, old_edge=None):
        if old_edge is None:
            old_edge = RequestCategoryTreeEdge.objects.filter(parent=parent_node, child=child_node).first()

        if old_edge is None:
            raise Exception(
                'edge from {} to {} cannot be found'.format(parent_node.category_id, child_node.category_id)
            )

        RequestCategoryTreeEdgeHistory.objects.create(
            history_action=Action.REMOVE.value,
            history_performed_at=performed_at,
            history_performed_by=performed_by,
            edge_id=old_edge.edge_id,
            parent_id=parent_node.category_id,
            child_id=child_node.category_id,
            meta_info=old_edge.meta_info,
        )

        self.tree.remove_edge(parent_node.category_id, child_node.category_id)

        old_edge.delete()


class CategoryTreeModificationHelper(object):
    """
    Possible test cases:
    - add root node / leaf node
    - modify node
    - remove root node / middle node / leaf node / copied node
    - remove root subtree / ordinary subtree / subtree containing a copied node
    - move a root node to any other place / a node to the top level / from one place to another
    - copy from one place to another

    Forbidden actions:
    - add to invalid node
    - modify invalid node
    - remove invalid root / ordinary node / ordinary node with invalid parent (None or invalid)
    - remove invalid root subtree / ordinary subtree / ordinary subtree with invalid parent (None or invalid)
    - move invalid node / from invalid parent / to invalid node
    - copy to the top level / a root node / to the same parent
    """

    def __init__(self, tree):
        self._db_helper = CategoryTreeDBHelper(tree)

    @property
    def tree(self):
        return self._db_helper.tree

    @tree.setter
    def tree(self, value):
        self._db_helper.tree = value

    def _make_node(self, meta_info, performed_at, performed_by):
        return self._db_helper.make_node(meta_info, performed_at, performed_by)

    def _alter_node(self, node, meta_info, performed_at, performed_by):
        return self._db_helper.alter_node(node, meta_info, performed_at, performed_by)

    def _destroy_node(self, node, performed_at, performed_by):
        return self._db_helper.destroy_node(node, performed_at, performed_by)

    def _make_edge(self, child_node, parent_node, performed_at, performed_by):
        return self._db_helper.make_edge(child_node, parent_node, performed_at, performed_by)

    def _alter_edge(self, edge, meta_info, performed_at, performed_by):
        return self._db_helper.alter_edge(edge, meta_info, performed_at, performed_by)

    def _destroy_edge(self, child_node, parent_node, performed_at, performed_by, *, old_edge=None):
        return self._db_helper.destroy_edge(child_node, parent_node, performed_at, performed_by, old_edge=old_edge)

    def add_node(self, parent_id, meta_info, performed_at, performed_by):
        self.tree.check_node_meta_info(meta_info)

        if parent_id is None:  # root node
            parent_node = None
        else:
            parent_node = RequestCategoryTreeNode.objects.filter(category_id=parent_id).first()

            if parent_node is None:
                raise Exception('node with id {} does not exist'.format(parent_id))

        with transaction.atomic(savepoint=False):
            node = self._make_node(meta_info, performed_at, performed_by)

            if parent_node is not None:
                self._make_edge(node, parent_node, performed_at, performed_by)

        return node.category_id

    def copy_node(self, node_id, new_parent_id, performed_at, performed_by):
        if new_parent_id is None:
            raise Exception('node copy cannot be a root node')

        if self.tree.exists(node_id) and self.tree.is_root_node(node_id):
            raise Exception('root node cannot be copied, only moved')

        # do not delete edge with parent
        self.move_node(node_id, None, new_parent_id, performed_at, performed_by)

    def move_node(self, node_id, old_parent_id, new_parent_id, performed_at, performed_by):
        if old_parent_id is not None and new_parent_id is not None and old_parent_id == new_parent_id:
            raise Exception('node cannot be moved to the same parent')

        node_id_collection = (node_id, old_parent_id, new_parent_id)
        nodes = RequestCategoryTreeNode.objects.filter(category_id__in=filter(None, node_id_collection))
        nodes_collection = {str(node.category_id): node for node in nodes}

        node, old_parent, new_parent = [nodes_collection.get(x, None) for x in node_id_collection]

        if node is None:
            raise Exception('node with id {} to be moved has not been found'.format(node_id))

        if old_parent_id is not None and old_parent is None:
            raise Exception('no edge exists between {} and {}'.format(node_id, old_parent_id))

        if new_parent_id is not None and new_parent is None:
            raise Exception('node with id {} selected as a new parent does not exist'.format(new_parent_id))

        with transaction.atomic(savepoint=False):
            if old_parent is not None:
                self._destroy_edge(node, old_parent, performed_at, performed_by)

            if new_parent is not None:
                self._make_edge(node, new_parent, performed_at, performed_by)

    def modify_node(self, node_id, meta_info, performed_at, performed_by):
        self.tree.check_node_meta_info(meta_info)

        node = RequestCategoryTreeNode.objects.filter(category_id=node_id).first()

        if node is None:
            raise Exception('node with id {} does not exist'.format(node_id))

        with transaction.atomic(savepoint=False):
            self._alter_node(node, meta_info, performed_at, performed_by)

    def remove_node(self, node_id, parent_id, performed_at, performed_by, *, node=None):
        if node is None:
            node = RequestCategoryTreeNode.objects.filter(category_id=node_id).first()

        if node is None:
            raise Exception('node with id {} does not exist'.format(node_id))

        all_parent_edges = list(RequestCategoryTreeEdge.objects.select_related('parent').filter(child=node))

        if parent_id is None:
            if all_parent_edges:
                raise Exception('parent_id can be omitted for a root node only')

            selected_parent_edges = []
        else:
            selected_parent_edges = [e for e in all_parent_edges if str(e.parent.category_id) == parent_id]

            if len(selected_parent_edges) != 1:
                raise Exception('no unique edge exists between {} and {}'.format(node_id, parent_id))

        child_edges = RequestCategoryTreeEdge.objects.select_related('child').filter(parent=node)

        with transaction.atomic(savepoint=False):
            for e in selected_parent_edges:
                self._destroy_edge(node, e.parent, performed_at, performed_by, old_edge=e)

            for e in child_edges:
                self._destroy_edge(e.child, node, performed_at, performed_by, old_edge=e)

            has_copy = (len(all_parent_edges) > 1)

            if not has_copy:
                self._destroy_node(node, performed_at, performed_by)

    def remove_subtree(self, node_id, parent_id, performed_at, performed_by):
        node = RequestCategoryTreeNode.objects.filter(category_id=node_id).first()

        if node is None:
            raise Exception('node with id {} does not exist'.format(node_id))

        all_parent_edges = list(RequestCategoryTreeEdge.objects.select_related('parent').filter(child=node))

        if parent_id is not None:
            parent_edges = [e for e in all_parent_edges if str(e.parent.category_id) == parent_id]

            if len(parent_edges) != 1:
                raise Exception('no unique edge exists between {} and {}'.format(node_id, parent_id))

            (edge, ) = parent_edges
            parent = edge.parent

        else:
            if all_parent_edges:
                raise Exception('node with id {} is not a root node, however parent has not been specified'
                                .format(node_id))

            parent = None
            edge = None

        current_nodes = [(node, parent, edge, False)]

        with transaction.atomic(savepoint=False):
            while current_nodes:
                current_node, current_node_parent, current_edge, visited = current_nodes.pop()

                _node_parents = list(self.tree.iter_parents(current_node.category_id))
                has_copy = (len(_node_parents) > 1)

                if visited:  # children has been processed already
                    if current_node_parent is not None:
                        self._destroy_edge(current_node, current_node_parent, performed_at, performed_by, old_edge=current_edge)

                    if not has_copy:
                        self._destroy_node(current_node, performed_at, performed_by)

                else:
                    current_nodes.append((current_node, current_node_parent, current_edge, True))

                    if not has_copy:
                        child_edges = RequestCategoryTreeEdge.objects.select_related('child').filter(parent=current_node)
                        children = ((e.child, current_node, e, False) for e in child_edges)
                        current_nodes.extend(children)
                    else:
                        # entire copied subtree should not be removed
                        pass
