import collections
import enum
import logging

from django.db import transaction

from cars.core.util import datetime_helper

from .selection_helper import CategoryTreeHistoryHelper, CategoryTreeSelectionHelper
from .modification_helper import CategoryTreeModificationHelper


LOGGER = logging.getLogger(__name__)


class CategoryTreeProcessingHelper(object):
    ActionDescription = collections.namedtuple('ActionDescription', ('action', 'data'))

    class ApiAction(enum.Enum):
        ADD = 'add'
        COPY = 'copy'
        MODIFY = 'modify'
        MOVE = 'move'
        REMOVE = 'remove'
        REMOVE_SUBTREE = 'remove_subtree'

    def __init__(self):
        self._selection_helper = CategoryTreeSelectionHelper()
        self._history_helper = CategoryTreeHistoryHelper()

        self._modification_helper = CategoryTreeModificationHelper(self.get_current_tree())

        self._action_mapping = {
            self.ApiAction.ADD.value: self._modification_helper.add_node,
            self.ApiAction.COPY.value: self._modification_helper.copy_node,
            self.ApiAction.MODIFY.value: self._modification_helper.modify_node,
            self.ApiAction.MOVE.value: self._modification_helper.move_node,
            self.ApiAction.REMOVE.value: self._modification_helper.remove_node,
            self.ApiAction.REMOVE_SUBTREE.value: self._modification_helper.remove_subtree,
        }

    def get_active_nodes(self, on_date):
        return list(self._selection_helper.iter_active_nodes(on_date))

    def get_formatted_history(self, on_date):
        return self._history_helper.get_formatted_history(on_date)

    def get_current_tree(self, node_filter=None, subtree_filter=None):
        return self._selection_helper.build_tree(node_filter=node_filter, subtree_filter=subtree_filter)

    def get_tree(self, on_date, node_filter=None, subtree_filter=None):
        return list(self._selection_helper.build_tree(on_date, node_filter, subtree_filter).iter_root_nodes())

    def perform_actions(self, action_descriptions, initiated_by):
        on_date = datetime_helper.utc_now()

        for action_description in action_descriptions:
            action_description = self.ActionDescription(**action_description)
            self._perform_action(action_description, on_date, initiated_by)

    def _perform_action(self, action_description, on_date, initiated_by):
        action_name = action_description.action
        action = self._action_mapping.get(action_name, None)

        if action is not None:
            try:
                action(performed_at=on_date, performed_by=initiated_by, **action_description.data)
            except Exception as exc:
                error_message = 'error applying tree modification: {}'.format(exc)
                LOGGER.exception(error_message)
                raise Exception(error_message)

        else:
            raise ValueError('unknown action: {}'.format(action_name))
