import logging

from cars.knowledge_base.models import (
    RequestCategoryTreeEdge,
    RequestCategoryTreeEdgeHistory,
    RequestCategoryTreeNode,
    RequestCategoryTreeNodeHistory,
)
from cars.knowledge_base.models.categories import Action

from .common_helper import merge_two_lists, Tree

LOGGER = logging.getLogger(__name__)


class CategoryTreeHistoryHelper(object):
    def get_raw_history(self, on_date):
        # latest entry is the last one
        edge_history = list(
            RequestCategoryTreeEdgeHistory.objects
            .filter(history_performed_at__gte=on_date)
            .order_by('-history_performed_at')
        )

        node_history = list(
            RequestCategoryTreeNodeHistory.objects
            .filter(history_performed_at__gte=on_date)
            .order_by('-history_performed_at')
        )

        history = merge_two_lists(edge_history, node_history, key=(lambda x: x.priority))
        return history

    def get_formatted_history(self, on_date):
        history = self.get_raw_history(on_date)
        formatted_history = [e.format_entry() for e in history]
        return formatted_history


class CategoryTreeSelectionHelper(object):
    ACTION_PRIORITY = {
        Action.REMOVE.value: 0,
        Action.MODIFY.value: 1,
        Action.ADD.value: 2,
    }

    def __init__(self):
        self._history_helper = CategoryTreeHistoryHelper()

    def iter_active_nodes(self, on_date=None):
        if on_date is None:
            actual_nodes = (
                {'id': x.category_id, 'meta_info': x.meta_info}
                for x in RequestCategoryTreeNode.objects.all()
            )
        else:
            actual_nodes = (
                {'id': x['id'], 'meta_info': x['meta_info']}
                for x in self.build_tree(on_date).iter_all_nodes()
            )

        return actual_nodes

    def build_tree(self, on_date=None, node_filter=None, subtree_filter=None):
        actual_tree = self._build_tree()

        if on_date is not None:
            self._rollback_tree(actual_tree, on_date)

        if node_filter is not None or subtree_filter is not None:
            self._filter_tree(actual_tree, node_filter, subtree_filter, node=None)

        return actual_tree

    def _build_tree(self):
        tree = Tree()

        # TODO: check request_category_edge_parent
        nodes = RequestCategoryTreeNode.objects.select_related().all()

        for node in nodes:
            tree.add_node(self._node_to_mapping(node))

        edges = RequestCategoryTreeEdge.objects.select_related().all()

        for edge in edges:
            parent_id, child_id = edge.parent.category_id, edge.child.category_id
            tree.add_edge(parent_id, child_id)

        return tree

    def _node_to_mapping(self, node):
        return Tree.make_node(node.category_id, node.meta_info)

    def _rollback_tree(self, actual_tree, on_date):
        history = self._history_helper.get_raw_history(on_date)

        for entry in history:
            if isinstance(entry, RequestCategoryTreeNodeHistory):
                self._rollback_node_history_entry(actual_tree, entry)
            elif isinstance(entry, RequestCategoryTreeEdgeHistory):
                self._rollback_edge_history_entry(actual_tree, entry)
            else:
                raise Exception('unknown entry type: {}'.format(type(entry)))

    def _rollback_node_history_entry(self, tree, node):
        assert isinstance(tree, Tree)
        if node.history_action == Action.REMOVE.value:
            tree.add_node(self._node_to_mapping(node))
        elif node.history_action == Action.MODIFY.value:
            tree.modify_node(self._node_to_mapping(node))
        elif node.history_action == Action.ADD.value:
            tree.remove_node(node.category_id)
        else:
            raise Exception('unknown action')

    def _rollback_edge_history_entry(self, tree, edge):
        assert isinstance(tree, Tree)
        if edge.history_action == Action.REMOVE.value:
            tree.add_edge(edge.parent_id, edge.child_id)
        elif edge.history_action == Action.ADD.value:
            tree.remove_edge(edge.parent_id, edge.child_id)
        else:
            raise Exception('unknown action')

    def _filter_tree(self, tree, node_filter, subtree_filter, node=None):
        filtered_tree = False

        nodes_to_check = list(tree.iter_root_nodes()) if node is None else [node]

        for node_to_check in nodes_to_check:
            filtered = subtree_filter(node_to_check) if subtree_filter is not None else True

            if filtered:
                filtered = node_filter(node_to_check) if node_filter is not None else True

                if not filtered and node_to_check['children']:
                    # explicitly filters all children to remove redundant
                    filtered = any([
                        self._filter_tree(tree, node_filter, subtree_filter, c)
                        for c in list(node_to_check['children'])
                    ])

            if not filtered:
                for parent_id in node_to_check['parent_ids']:
                    tree.remove_edge(parent_id, node_to_check['id'])

                tree.remove_node(node_to_check['id'])

            filtered_tree = filtered_tree or filtered

        return filtered_tree
