import enum
import uuid

from django.contrib.postgres.fields import JSONField
from django.db import models

from cars.users.models import User
from cars.users.serializers import UserSerializer


class Action(enum.Enum):
    ADD = 'a'
    MODIFY = 'm'
    REMOVE = 'r'


ACTION_PRIORITY = {
    Action.REMOVE.value: 0,
    Action.MODIFY.value: 1,
    Action.ADD.value: 2,
}


class RequestCategoryTreeNode(models.Model):
    category_id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    meta_info = JSONField(default=dict)

    class Meta:
        db_table = 'request_category_tree_node'
        indexes = []

    @property
    def labels(self):
        return self.meta_info.get('labels', [])

    @labels.setter
    def labels(self, value):
        self.meta_info['labels'] = value


class RequestCategoryTreeNodeHistory(models.Model):
    history_id = models.AutoField(primary_key=True)

    history_action = models.CharField(
        max_length=1,
        choices=[(x.value, x.name) for x in Action],
    )

    history_performed_at = models.DateTimeField()

    history_performed_by = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
    )

    category_id = models.UUIDField()

    meta_info = JSONField()

    class Meta:
        db_table = 'request_category_tree_node_history'
        indexes = [
            models.Index(
                fields=['category_id'],
                name='request_category_tree_node_history_category_id_idx',
            ),
        ]

    @property
    def priority(self):
        # revert order: remove node (0), remove edge (1), modify node/edge (2), add edge (3), add node (4)
        return -self.history_performed_at.timestamp(), 2 * ACTION_PRIORITY[self.history_action]

    def format_entry(self):
        performed_by = UserSerializer(self.history_performed_by).data if self.history_performed_by is not None else None
        return {
            'type': 'node',
            'action': Action(self.history_action).name,
            'performed_by': performed_by,
            'performed_at': self.history_performed_at.timestamp(),
            'node_id': str(self.category_id),
            'meta_info': self.meta_info,
        }

    def __str__(self):
        return '<RequestCategoryTreeNodeHistory: {}>'.format(self.format_entry())


class RequestCategoryTreeEdge(models.Model):
    edge_id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    parent = models.ForeignKey(
        RequestCategoryTreeNode,
        on_delete=models.CASCADE,
        related_name='request_category_tree_edge_parent',
    )

    child = models.ForeignKey(
        RequestCategoryTreeNode,
        on_delete=models.CASCADE,
        related_name='request_category_tree_edge_child',
    )

    meta_info = JSONField(default=dict)

    class Meta:
        db_table = 'request_category_tree_edge'
        indexes = []


class RequestCategoryTreeEdgeHistory(models.Model):
    history_id = models.AutoField(primary_key=True)

    history_action = models.CharField(
        max_length=1,
        choices=[(x.value, x.name) for x in Action],
    )

    history_performed_at = models.DateTimeField()

    history_performed_by = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
    )

    edge_id = models.UUIDField()

    parent_id = models.UUIDField()

    child_id = models.UUIDField()

    meta_info = JSONField()

    class Meta:
        db_table = 'request_category_tree_edge_history'
        indexes = [
            models.Index(
                fields=['edge_id'],
                name='request_category_tree_edge_history_edge_id_idx',
            ),
        ]

    @property
    def priority(self):
        # revert order: remove node (0), remove edge (1), modify node/edge (2), add edge (3), add node (4)
        return -self.history_performed_at.timestamp(), 1 + ACTION_PRIORITY[self.history_action]

    def format_entry(self):
        performed_by = UserSerializer(self.history_performed_by).data if self.history_performed_by is not None else None
        return {
            'type': 'edge',
            'action': Action(self.history_action).name,
            'performed_by': performed_by,
            'performed_at': self.history_performed_at.timestamp(),
            'edge_id': str(self.edge_id),
            'meta_info': self.meta_info,
            'parent_id': str(self.parent_id),
            'child_id': str(self.child_id),
        }

    def __str__(self):
        return '<RequestCategoryTreeEdgeHistory: {}>'.format(self.format_entry())
