import itertools

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_500_INTERNAL_SERVER_ERROR

from cars.core.util import datetime_helper
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory

from ..core import CategoryTreeProcessingHelper
from ..serializers.category_tree import TreeViewArgumentsSerializer, TreeHistoryViewArgumentsSerializer


class CategoryAPIView(DrivePermissionAPIView):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = CategoryTreeProcessingHelper()


class AllCategoriesAPIView(CategoryAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    arguments_serializer_class = TreeViewArgumentsSerializer

    def do_get(self, request):
        on_date = request.arguments.get('on_date', datetime_helper.utc_now())
        response_entries = self._helper.get_active_nodes(on_date)
        return Response(
            {
                'entries_count': len(response_entries),
                'data': response_entries,
            }
        )


class CategoriesTreeHistoryAPIView(CategoryAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    arguments_serializer_class = TreeHistoryViewArgumentsSerializer

    def do_get(self, request):
        on_date = request.arguments['since']
        response_entries = self._helper.get_formatted_history(on_date)
        return Response(
            {
                'entries_count': len(response_entries),
                'data': response_entries,
            }
        )


class CategoriesTreeAPIView(CategoryAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    arguments_serializer_class = TreeViewArgumentsSerializer

    def do_get(self, request):
        on_date = request.arguments.get('on_date', datetime_helper.utc_now())

        text_filter = request.arguments['filter'].lower()
        if text_filter:
            node_filter = (
                lambda n: any(
                    (text_filter in l.lower()) for l in
                    itertools.chain(n['meta_info']['labels'].values(), n['meta_info'].get('keywords', []))
                )
            )
        else:
            node_filter = None

        enabled_only = request.arguments['enabled_only']

        origin = request.arguments.get('origin', None)

        if origin is None:
            direction = None
        elif origin in (
                'CARSHARING', 'CARSHARING_VIP', 'AUDIOTELE_INCOMING',
                'CC_INTERNAL_REALTIME_INCOMING', 'AUDIOTELE_REALTIME_INCOMING'
        ):
            direction = 'incoming'
        elif origin in ('AUDIOTELE_OUTGOING', 'CC_INTERNAL_ALTAY_OUTGOING'):
            direction = 'outgoing'
        else:
            return Response({'errors': 'incorrect direction'}, status=HTTP_400_BAD_REQUEST)

        subtree_filter = (lambda n: (
            (not enabled_only or n['meta_info'].get('enabled', True)) and
            (direction is None or n['meta_info'].get('call_direction') == direction)
        ))

        response_tree = self._helper.get_tree(on_date, node_filter, subtree_filter)
        return Response(
            {
                'children': response_tree,
            }
        )


class CategoriesTreeActionPerformAPIView(CategoryAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    def do_post(self, request):
        initiated_by = request.user
        action_descriptions = request.data.get('data', [])
        try:
            self._helper.perform_actions(action_descriptions, initiated_by)
        except ValueError as exc:
            return Response(str(exc), status=HTTP_400_BAD_REQUEST)
        except Exception as exc:
            return Response(str(exc), status=HTTP_500_INTERNAL_SERVER_ERROR)
        return Response()
