import base64
import os

import furl
from django.utils import timezone

import cars.settings
from cars.carsharing.models import CarsharingAcceptancePhoto


class CarsharingAcceptancePhotoManager:

    def __init__(self, order_item, mds_client, mds_bucket, images_access_host):
        self._order_item = order_item
        self._mds_client = mds_client
        self._mds_bucket = mds_bucket
        self._host_furl = furl.furl(images_access_host)

    @classmethod
    def from_settings(cls, order_item, mds_client):
        return cls(
            order_item=order_item,
            mds_client=mds_client,
            mds_bucket=cars.settings.CARSHARING['acceptance']['mds_bucket_name'],
            images_access_host=cars.settings.CARSHARING['acceptance']['images_access_host'],
        )

    def add_acceptance_photo(self, id, b64content, mime_type):
        if CarsharingAcceptancePhoto.objects.filter(id=id).first() is not None:
            # Duplicate, don't upload
            return

        photo = CarsharingAcceptancePhoto(
            id=id,
            acceptance=self._order_item.carsharing_acceptance,
            submitted_at=timezone.now(),
            mime_type=mime_type,
            link=self.get_photo_link(id),
        )

        content = base64.b64decode(b64content.encode('utf-8'))
        self._mds_client.put_object(
            bucket=self._mds_bucket,
            key=self.get_photo_mds_key(id),
            body=content,
        )

        photo.save()

    def get_acceptance_photo(self, photo_obj):
        photo_content = self._mds_client.get_object_content(
            bucket=self._mds_bucket,
            key=self.get_photo_mds_key(photo_obj.id),
        )
        return photo_content

    def get_photo_mds_key(self, photo_id):
        return os.path.join(
            str(self._order_item.order.user.id),
            str(self._order_item.id),
            str(photo_id),
        )

    def get_photo_link(self, photo_id):
        return self._host_furl.join(self.get_photo_mds_key(photo_id)).url
