import datetime
import logging
import time

import pytz
from django.utils import timezone

import cars.settings
from cars.core.util import import_class
from ..core.order_request import OrderItemRequest
from ..core.push_client import PUSH_CLIENT
from ..models.order_item import OrderItem
from ..models.order_item_tariff import OrderItemTariff, PerMinuteOrderItemTariffParams
from .order_manager import OrderManager
from .order_item_requests.carsharing import (
    CarsharingParkingOrderItemRequest,
    CarsharingReservationPaidOrderItemRequest,
)


LOGGER = logging.getLogger(__name__)


class CarsharingFreeParkingTracker:

    calculator = import_class(cars.settings.CALCULATOR['client']['class']).from_settings()

    def __init__(self, order_manager):
        self._order_manager = order_manager

    @classmethod
    def from_settings(cls):
        return cls(
            order_manager=OrderManager.from_settings(push_client=PUSH_CLIENT),
        )

    def track_all(self):
        items = (
            OrderItem.objects
            .with_related()
            .filter(
                finished_at__isnull=True,
                type__in=[
                    OrderItem.Type.CARSHARING_RESERVATION_PAID.value,
                    OrderItem.Type.CARSHARING_PARKING.value,
                ],
            )
        )

        for item in items:
            try:
                self.track_item(item)
            except Exception:
                LOGGER.exception('failed to track free parking for item %s', item.id)
                continue

    def track_item(self, item):
        if item.tariff.get_type() is not OrderItemTariff.Type.PER_MINUTE:
            return

        user = item.order.user
        car = item.get_impl().car
        current_tariff = self.calculator.get_carsharing_base_tariffs(
            user=user,
            cars=[car],
            tz=pytz.UTC,
        )[car]

        should_be_free = current_tariff.parking_cost_per_minute == 0
        is_free = item.tariff.per_minute_params.cost_per_minute == 0

        if item.order.has_plus_discount:
            current_time = time.time() % 86400
            if current_time <= 3600 * 3:
                should_be_free = True

        start_date = self._get_tariff_start_date(
            order=item.order,
            start_time=current_tariff.start_time,
        )
        start_date = min(start_date, timezone.now())
        start_date = max(start_date, item.started_at)

        if should_be_free and not is_free:
            self._start_free_period(
                item=item,
                car=car,
                start_date=start_date,
            )
        elif is_free and not should_be_free:
            snapshot = item.order.get_tariff_snapshot()
            if snapshot is not None:
                item_type = item.get_type()
                if item_type is OrderItem.Type.CARSHARING_RESERVATION_PAID:
                    tariff = snapshot.carsharing_reservation_paid
                elif item_type is OrderItem.Type.CARSHARING_PARKING:
                    tariff = snapshot.carsharing_parking
                else:
                    raise RuntimeError('unreachable: {}'.format(item_type))
                assert tariff.get_type() is OrderItemTariff.Type.PER_MINUTE
                cost_per_minute = tariff.per_minute_params.cost_per_minute
                if cost_per_minute == 0:
                    cost_per_minute = current_tariff.parking_cost_per_minute
            else:
                cost_per_minute = current_tariff.parking_cost_per_minute

            self._finish_free_period(
                item=item,
                car=car,
                cost_per_minute=cost_per_minute,
                start_date=start_date,
            )

    def _start_free_period(self, item, car, start_date):
        LOGGER.info(
            'starting free period for order %s car %s from %s',
            item.order.id,
            car.id,
            start_date,
        )
        self._switch_tariff(item=item, car=car, cost_per_minute=0, start_date=start_date)

    def _finish_free_period(self, item, car, cost_per_minute, start_date):
        LOGGER.info(
            'finishing free period for order %s car %s at %s minute cost from %s',
            item.order.id,
            car.id,
            cost_per_minute,
            start_date,
        )
        self._switch_tariff(
            item=item, car=car,
            cost_per_minute=cost_per_minute,
            start_date=start_date,
        )

    def _switch_tariff(self, item, car, cost_per_minute, start_date):
        item_type = item.get_type()
        if item_type is OrderItem.Type.CARSHARING_RESERVATION_PAID:
            request_impl = CarsharingReservationPaidOrderItemRequest(
                user=item.order.user,
                car_id=car.id,
            )
        elif item_type is OrderItem.Type.CARSHARING_PARKING:
            request_impl = CarsharingParkingOrderItemRequest(
                user=item.order.user,
                car_id=car.id,
            )
        else:
            raise RuntimeError('unreachable: {}'.format(item_type))

        request = OrderItemRequest(
            user=item.order.user,
            item_type=item_type,
            impl=request_impl,
            tariff=OrderItemTariff(
                type=OrderItemTariff.Type.PER_MINUTE.value,
                per_minute_params=PerMinuteOrderItemTariffParams.objects.create(
                    cost_per_minute=cost_per_minute,
                ),
            ),
        )

        self._order_manager.add_order_item(
            order=item.order,
            order_item_request=request,
            started_at=start_date,
        )

    def _get_tariff_start_date(self, order, start_time):
        if start_time is None:
            start_time = datetime.time(0, 0)

        item = order.get_sorted_items()[-1]
        dt = item.started_at

        if dt.time() < start_time:
            base_date = dt
        else:
            base_date = timezone.now()

        start_date = self._replace_time(dt=base_date, time=start_time)

        return start_date

    def _replace_time(self, dt, time):
        return dt.replace(
            hour=time.hour,
            minute=time.minute,
            second=time.second,
            microsecond=time.microsecond,
        )
