import datetime
import logging

from django.db import transaction

from cars.core.pusher import BasePusher
from ..core.order_item_managers.carsharing_reservation_paid import CarsharingReservationPaidManager
from ..core.order_item_requests.carsharing import CarsharingReservationPaidOrderItemRequest
from ..core.order_request import OrderItemRequest, OrderItemRequestContext
from ..core.push_client import PUSH_CLIENT
from ..models.order import Order
from ..models.order_item import OrderItem
from .order_manager import OrderManager


LOGGER = logging.getLogger(__name__)


class CarsharingReservationTerminator:

    def __init__(self, order_manager, pusher):
        self._order_manager = order_manager
        self._pusher = pusher

    @classmethod
    def from_settings(cls):
        return cls(
            order_manager=OrderManager.from_settings(push_client=PUSH_CLIENT),
            pusher=BasePusher.from_settings(),
        )

    def process_ongoing_reservations(self):
        reservation_items = (
            OrderItem.objects
            .select_related('carsharing_reservation')
            .filter(
                type=OrderItem.Type.CARSHARING_RESERVATION.value,
                finished_at__isnull=True,
            )
        )
        for item in reservation_items:
            try:
                self.process_reservation(item)
            except Exception:
                LOGGER.exception('failed to check a reservation for termination: %s', item.id)
                continue

    def process_reservation(self, reservation_item):
        assert reservation_item.get_type() is OrderItem.Type.CARSHARING_RESERVATION

        if reservation_item.finished_at is not None:
            return

        reservation = reservation_item.carsharing_reservation
        max_duration = datetime.timedelta(
            seconds=reservation.max_duration_seconds,
        )

        if reservation_item.duration >= max_duration:
            self._make_reservation_paid(reservation_item)

    def _make_reservation_paid(self, reservation_item):
        with transaction.atomic():
            Order.objects.select_for_update().get(id=reservation_item.order_id)
            reservation_item = (
                OrderItem.objects
                .select_related(
                    'carsharing_reservation',
                    'order__user',
                )
                .get(id=reservation_item.id)
            )
            request = self._make_reservation_paid_request(reservation_item=reservation_item)
            self._order_manager.add_order_item(
                order=reservation_item.order,
                order_item_request=request,
            )

    def _make_reservation_paid_request(self, reservation_item):
        assert reservation_item.get_type() is OrderItem.Type.CARSHARING_RESERVATION

        request_impl = CarsharingReservationPaidOrderItemRequest(
            user=reservation_item.order.user,
            car_id=reservation_item.carsharing_reservation.car_id,
        )
        request = OrderItemRequest(
            user=reservation_item.order.user,
            item_type=OrderItem.Type.CARSHARING_RESERVATION_PAID,
            impl=request_impl,
            tariff=CarsharingReservationPaidManager.pick_from_order_item_request(
                user=reservation_item.order.user,
                request_impl=request_impl,
                context=OrderItemRequestContext(
                    oauth_token=None,
                ),
            ),
        )

        return request
