import collections
import decimal
import logging

from django.db.models import Prefetch
from django.utils import timezone

import cars.settings
from cars.billing.iface.payment import IPayment
from cars.users.core.user_profile_updater import UserProfileUpdater
from cars.users.models.user import User
from ..core.order_payment_processor import OrderPaymentProcessor
from ..models.order import Order
from ..models.order_item_payment import OrderItemPayment
from ..models.order_payment_method import OrderPaymentMethod


LOGGER = logging.getLogger(__name__)


class OrderDebtManager:

    class Error(Exception):
        pass

    class InsufficientFundsError(Error):
        pass

    def __init__(self,
                 order_payment_processor,
                 debt_order_termination_threshold,
                 debt_order_termination_delay):
        self._order_payment_processor = order_payment_processor
        self._debt_order_termination_threshold = debt_order_termination_threshold
        self._debt_order_termination_delay = debt_order_termination_delay

    @classmethod
    def from_settings(cls, push_client):  # pylint: disable=unused-argument
        settings = cars.settings.ORDERS
        return cls(
            order_payment_processor=OrderPaymentProcessor.from_settings(),
            debt_order_termination_threshold=settings['debt_order_termination']['threshold'],
            debt_order_termination_delay=settings['debt_order_termination']['delay'],
        )

    @property
    def debt_order_termination_delay(self):
        return self._debt_order_termination_delay

    @property
    def debt_order_termination_threshold(self):
        return self._debt_order_termination_threshold

    def get_debt(self, user, respect_user_status=True):
        if respect_user_status and user.get_status() is not User.Status.DEBT:
            return decimal.Decimal(0)

        debt_orders = (
            Order.objects
            .with_payments()
            .filter(
                user=user,
                payment_status__in=[
                    Order.PaymentStatus.NEW.value,
                    Order.PaymentStatus.ERROR.value,
                ]
            )
        )
        debt = sum(self._get_order_debt_amount(order) for order in debt_orders)

        return debt

    def mark_debtors(self):
        debt_per_user = self.get_debt_per_user()

        for user, debt in debt_per_user.items():
            if debt <= 0:
                continue
            UserProfileUpdater(user).update_status(User.Status.DEBT)

    def get_debt_per_user(self, only_active=True):
        maybe_debt_orders = (
            Order.objects
                .with_payments()
                .filter(
                payment_status__in=[
                    Order.PaymentStatus.NEW.value,
                    Order.PaymentStatus.ERROR.value,
                ],
            )
        )
        if only_active:
            maybe_debt_orders = maybe_debt_orders.filter(user__status=User.Status.ACTIVE.value)

        debt_per_user = collections.defaultdict(lambda: 0)
        for order in maybe_debt_orders:
            debt_per_user[order.user] += self._get_order_debt_amount(order)

        return debt_per_user

    def unmark_debtors(self):
        users = (
            User.objects
            .filter(status=User.Status.DEBT.value)
            .prefetch_related(
                Prefetch(
                    'orders',
                    queryset=Order.objects.filter(
                        payment_status__in=[
                            Order.PaymentStatus.NEW.value,
                            Order.PaymentStatus.ERROR.value,
                        ]
                    ),
                    to_attr='maybe_error_orders',
                )
            )
        )
        for user in users:
            if user.maybe_error_orders or user.status == User.Status.BLOCKED.value:
                continue
            LOGGER.info('unmarking debtor %s', user.id)
            UserProfileUpdater(user).update_status(User.Status.ACTIVE)

    def try_unmark_debtor(self, user):
        if user.get_status() is User.Status.ACTIVE:
            return user

        error_orders = (
            Order.objects
            .filter(
                user=user,
                payment_status=Order.PaymentStatus.ERROR.value,
            )
        )

        if not error_orders.exists() and user.status == User.Status.DEBT.value:
            LOGGER.info('unmarking debtor %s', user.id)
            UserProfileUpdater(user).update_status(User.Status.ACTIVE)

        return user

    def update_old_backend_debt_records(self):
        # mark debtors
        debt_records = self.get_debt_per_user(only_active=False)
        for user, debt_value in debt_records.items():
            user_debt = decimal.Decimal(str(debt_value))
            if user_debt != user.old_backend_debt:
                LOGGER.info(
                    'updating old_backend_debt for %s from %s to %s',
                    str(user.id),
                    str(user.old_backend_debt),
                    str(user_debt)
                )
                user.old_backend_debt = user_debt
                user.save()

        # unmark debtors
        users = (
            User.objects
            .filter(status=User.Status.DEBT.value)
            .prefetch_related(
                Prefetch(
                    'orders',
                    queryset=Order.objects.filter(
                        payment_status__in=[
                            Order.PaymentStatus.NEW.value,
                            Order.PaymentStatus.ERROR.value,
                        ]
                    ),
                    to_attr='maybe_error_orders',
                )
            )
        )
        for user in users:
            if user.maybe_error_orders:
                continue
            user.old_backend_debt = None
            LOGGER.info('dropping debt for %s', str(user.id))
            user.save()

    def _get_order_debt_amount(self, order):
        progress = self._order_payment_processor.get_order_payment_progress(order)
        if progress.has_errors:
            debt_amount = progress.cost - (progress.paid_amount + progress.in_progress_amount)
        else:
            debt_amount = 0
        return debt_amount

    def check_if_terminate_order_for_debt(self, user):
        debt = self.get_debt(user)
        return debt > self._debt_order_termination_threshold

    def retry_all_debts(self, policy):
        error_orders = (
            Order.objects
            .with_payments()
            .filter(
                completed_at__isnull=False,
                payment_status=Order.PaymentStatus.ERROR.value,
            )
        )

        for order in error_orders:
            if not policy.should_retry(order):
                continue
            LOGGER.info('retrying payments for order %s', order.id)
            self._make_payments_with_active_payment_method(order)

    def repay_debt(self, user, timeout):
        if user.get_status() is not User.Status.DEBT:
            return

        error_orders = (
            Order.objects
            .with_payments()
            .filter(
                user=user,
                payment_status=Order.PaymentStatus.ERROR.value,
            )
        )

        all_payments = []
        for order in error_orders:
            payments = self._make_payments_with_active_payment_method(order)
            all_payments.extend(payments)

        self._order_payment_processor.wait_for_payments(
            payments=all_payments,
            timeout=timeout,
        )

        updated_error_orders = (
            Order.objects
            .with_payments()
            .filter(id__in=[o.id for o in error_orders])
        )
        for order in updated_error_orders:
            self._order_payment_processor.finalize_order_payment_status(order=order)

        self.try_unmark_debtor(user=user)

        self._raise_for_repay_debt_payments(user=user, payments=all_payments)

    def _make_payments_with_active_payment_method(self, order):
        payment_method = OrderPaymentMethod(
            type=OrderPaymentMethod.Type.CARD,
            card_paymethod_id=order.user.get_credit_card().paymethod_id,
        )
        payments = self._order_payment_processor.make_payments(
            order=order,
            payment_method=payment_method,
        )
        return payments

    def _raise_for_repay_debt_payments(self, user, payments):
        if user.get_status() is not User.Status.DEBT:
            return

        for payment in payments:
            if payment.get_payment_method() is not OrderItemPayment.PaymentMethod.CARD:
                continue
            if payment.card_payment.resp_code == 'not_enough_funds':
                raise self.InsufficientFundsError

        raise self.Error


class DebtRetryPolicy:

    def __init__(self, retry_interval):
        self._retry_interval = retry_interval

    @classmethod
    def from_settings(cls):
        return DebtRetryPolicy(
            retry_interval=cars.settings.ORDERS['debt_retry_interval'],
        )

    def should_retry(self, order):
        assert order.get_payment_status() is Order.PaymentStatus.ERROR

        has_in_progress_payments = False
        last_error_payment_created_at = None

        for item in order.items.all():
            for payment in item.payments.all():
                status = payment.get_impl().get_generic_status()

                if status is IPayment.GenericStatus.IN_PROGRESS:
                    has_in_progress_payments = True
                    break

                if status is not IPayment.GenericStatus.ERROR:
                    continue

                if (last_error_payment_created_at is None
                        or payment.created_at > last_error_payment_created_at):
                    last_error_payment_created_at = payment.created_at

        if has_in_progress_payments:
            should_retry = False
        elif last_error_payment_created_at is None:
            LOGGER.info('should retry debt for order %s: no error payments', order.id)
            should_retry = True
        elif timezone.now() - last_error_payment_created_at > self._retry_interval:
            LOGGER.info(
                'should retry debt for order %s: last retried at %s',
                order.id,
                last_error_payment_created_at,
            )
            should_retry = True
        else:
            should_retry = False

        return should_retry
