import logging

from django.utils import timezone

from ..models.order_item import OrderItem
from .order_manager import OrderManager
from .push_client import PUSH_CLIENT


LOGGER = logging.getLogger(__name__)


class OrderDebtTerminator:

    def __init__(self, order_manager):
        self._order_manager = order_manager

    @classmethod
    def from_settings(cls):
        return cls(
            order_manager=OrderManager.from_settings(push_client=PUSH_CLIENT),
        )

    def process_all(self):
        items = (
            OrderItem.objects
            .select_related('carsharing_parking')
            .filter(
                type=OrderItem.Type.CARSHARING_PARKING.value,
                finished_at__isnull=True,
                carsharing_parking__debt_termination_max_duration_seconds__isnull=False,
            )
        )

        for item in items:
            try:
                self._process_parking(item)
            except Exception:
                LOGGER.exception('debt termination attempt for order %s failed', item.order_id)
                continue

    def _process_parking(self, item):
        seconds_passed = (timezone.now() - item.started_at).total_seconds()
        if seconds_passed <= item.carsharing_parking.debt_termination_max_duration_seconds:
            return

        LOGGER.info('terminating order %s for debt', item.order_id)
        self._order_manager.send_action(
            order_item=item,
            action='finish_debt',
            context=None,
        )
