import datetime
import logging
import os.path

from django.db.models import Q

import cars.settings
from cars.core.util import make_yt_client
from ..models.order import Order
from ..models.order_item_tariff import OrderItemTariff
from .order_payment_processor import OrderPaymentProcessor


LOGGER = logging.getLogger(__name__)


class OrderExportBuilder:

    def __init__(self, order_payment_processor):
        self._order_payment_processor = order_payment_processor

    @classmethod
    def from_settings(cls):
        return cls(
            order_payment_processor=OrderPaymentProcessor.from_settings(),
        )

    def update_yt_export(self, yt_manager):
        first_order = (
            Order.objects
            .filter(completed_at__isnull=False)
            .order_by('completed_at')
            .first()
        )
        if first_order is None:
            LOGGER.info('no completed orders')
            return

        first_order_date = first_order.completed_at.date()
        exported_dates = yt_manager.get_exported_dates()

        date = first_order_date
        today = datetime.date.today()
        yesterday = today - datetime.timedelta(days=1)

        while True:
            if date not in [today, yesterday]:  # changed due to backends switch crutch; if smth weird happens, feel free to roll back
                date += datetime.timedelta(days=1)
                continue

            if date > today:
                break

            export = self.build_for_date(date)
            if not export:
                continue

            LOGGER.info('uploading %s export to yt', date)
            yt_manager.upload_export_for_date(date, export)

            date += datetime.timedelta(days=1)

    def build_for_date(self, date):
        orders = (
            Order.objects
            .with_related()
            .using(cars.settings.DB_RO_ID)
            .filter(
                Q(completed_at__date=date)
                |
                Q(created_at__date__gte=date, completed_at__isnull=True)
            )
            .order_by('completed_at')
        )
        return self.build_for_orders(orders)

    def build_for_orders(self, orders):
        data = []
        for order in orders:
            order_data = self._order_to_dict(order)
            data.append(order_data)
        return data

    def _order_to_dict(self, order):
        items = []
        for item in order.get_sorted_items():
            item_data = self._order_item_to_dict(item)
            items.append(item_data)

        data = {
            'id': str(order.id),
            'created_at': order.created_at.timestamp(),
            'completed_at': None if order.completed_at is None else order.completed_at.timestamp(),
            'created_at_isoformat': order.created_at.isoformat(),
            'completed_at_isoformat': (
                None if order.completed_at is None else order.completed_at.isoformat()
            ),
            'user': {
                'id': str(order.user_id),
            },
            'cost': {
                'value': str(self._order_payment_processor.get_order_cost(order)),
            },
            'items': items,
        }

        return data

    def _order_item_to_dict(self, order_item):
        car = order_item.get_impl().car

        payments_data = []
        for payment in order_item.payments.all():
            payment_data = self._payment_to_dict(payment)
            payments_data.append(payment_data)

        data = {
            'id': str(order_item.id),
            'type': order_item.type,
            'started_at': order_item.started_at.timestamp(),
            'finished_at': (
                None if order_item.finished_at is None else order_item.finished_at.timestamp()
            ),
            'started_at_isoformat': order_item.started_at.isoformat(),
            'finished_at_isoformat': (
                None if order_item.finished_at is None else order_item.finished_at.isoformat()
            ),
            'tariff': self._order_item_tariff_to_dict(order_item.tariff),
            'cost': {
                'value': str(self._order_payment_processor.get_order_item_cost(order_item)),
            },
            'payments': payments_data,
            'params': {
                'car': {
                    'id': str(car.id),
                    'number': car.number,
                },
            },
        }

        return data

    def _order_item_tariff_to_dict(self, tariff):
        if tariff is None:
            return None

        tariff_type = tariff.get_type()
        if tariff_type is OrderItemTariff.Type.FIX:
            params = {
                'cost': str(tariff.fix_params.cost),
            }
        elif tariff_type is OrderItemTariff.Type.PER_MINUTE:
            params = {
                'cost_per_minute': str(tariff.per_minute_params.cost_per_minute),
            }
        else:
            raise RuntimeError('unreachable {}'.format(tariff_type))

        data = {
            'type': tariff.type,
            'params': params,
        }

        return data

    def _payment_to_dict(self, payment):
        impl = payment.get_impl()
        data = {
            'id': str(payment.id),
            'payment_method': payment.payment_method,
            'created_at': payment.created_at.timestamp(),
            'created_at_isoformat': payment.created_at.isoformat(),
            'amount': str(payment.amount),
            'status': impl.get_generic_status().value,
        }
        return data


class OrderExportYtManager:

    def __init__(self, yt_client, export_path):
        self._yt_client = yt_client
        self._root = export_path

    @classmethod
    def from_settings(cls):
        yt_client = make_yt_client('data')
        return cls(
            yt_client=yt_client,
            export_path=cars.settings.ORDERS['export']['path'],
        )

    def setup(self):
        self._ensure_root_dir()

    def get_export_path_for_date(self, date):
        assert isinstance(date, datetime.date)
        return os.path.join(self._root, date.isoformat())

    def get_exported_dates(self):
        dates = set()
        for table in self._yt_client.list(self._root):
            try:
                date = datetime.datetime.strptime(table, '%Y-%m-%d').date()
            except ValueError:
                LOGGER.warning('unexpected export table name: %s', table)
                continue
            dates.add(date)
        return dates

    def upload_export_for_date(self, date, export):
        path = self.get_export_path_for_date(date)
        self._yt_client.write_table(path, export)

    def _ensure_root_dir(self):
        if not self._yt_client.exists(self._root):
            LOGGER.info('creating missing root directory %s', self._root)
            self._yt_client.mkdir(self._root, recursive=True)
