import collections
import logging

from django.utils import timezone

import cars.settings
from cars.core.util import import_class
from ...models.order_item_tariff import (
    FixOrderItemTariffParams, OrderItemTariff, PerMinuteOrderItemTariffParams,
)
from ...models.order_tariff_snapshot import OrderTariffSnapshot


LOGGER = logging.getLogger(__name__)


class BaseOrderItemManager:

    class ActionError(Exception):

        def __init__(self, *args, code, **kwargs):
            super().__init__(*args, **kwargs)
            self.code = code

        def __repr__(self):
            return self.code

    class InvalidActionError(Exception):
        pass

    @classmethod
    def _finish_item_impl(cls, item, finished_at=None):
        if item.finished_at is not None:
            LOGGER.warning('trying to finish order item twice: %s', item.id)
            return

        if finished_at is None:
            item.finished_at = timezone.now()
        else:
            if finished_at < item.started_at:
                raise RuntimeError
            item.finished_at = finished_at
        try:
            item.save()
        except Exception:
            item.refresh_from_db()
            raise

    @classmethod
    def _finish_item(cls, item, finished_at=None):
        from .factory import OrderItemManagerFactory
        manager = OrderItemManagerFactory.get_class_from_item_type(item.get_type())
        manager._finish_item_impl(item, finished_at)

    def __init__(self, order_item):
        self._item = order_item

    @property
    def item(self):
        return self._item

    @classmethod
    def get_item_request_class(cls):
        raise NotImplementedError

    @classmethod
    def get_item_type(cls):
        raise NotImplementedError

    @classmethod
    def _get_tariff_picker(cls):
        raise NotImplementedError

    def send_action(self, action, context, params=None):
        raise NotImplementedError

    @classmethod
    def prepare_request(cls, request):
        request.prepare()

    @classmethod
    def materialize_request(cls, order, request, started_at=None):
        return request.materialize(order=order, started_at=started_at)

    @classmethod
    def pick_from_order_item_request(cls, user, request_impl, context):
        return cls._get_tariff_picker().pick_from_order_item_request(
            user=user,
            request_impl=request_impl,
            context=context,
        )

    @classmethod
    def pick_from_order_request(cls, request):
        return cls._get_tariff_picker().pick_from_order_request(request)


class BaseTariffPicker:

    calculator = import_class(cars.settings.CALCULATOR['client']['class']).from_settings()

    def pick_from_order_request(self, request):
        raise NotImplementedError

    def pick_from_order_item_request(self, user, request_impl, context):
        raise NotImplementedError

    def _build_fix_tariff(self, cost):
        tariff = OrderItemTariff(
            type=OrderItemTariff.Type.FIX.value,
            fix_params=FixOrderItemTariffParams(
                cost=cost,
            ),
        )
        return tariff

    def _build_per_minute_tariff(self, cost_per_minute):
        tariff = OrderItemTariff(
            type=OrderItemTariff.Type.PER_MINUTE.value,
            per_minute_params=PerMinuteOrderItemTariffParams(
                cost_per_minute=cost_per_minute,
            ),
        )
        return tariff

    def _get_current_order_tariff_snapshot(self, user):
        return (
            OrderTariffSnapshot.objects
            .with_related()
            .filter(
                order__user=user,
                order__completed_at__isnull=True,
            )
            .first()
        )


OrderItemActionContext = collections.namedtuple(
    'OrderItemActionContext',
    [],
)
