import enum
import logging

from django.db import transaction

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.acceptance import CarsharingAcceptance
from cars.carsharing.models.car import Car
from cars.core.util import import_class
from ...models.order_item import OrderItem
from ...models.order_item_tariff import OrderItemTariff, FixOrderItemTariffParams
from ..acceptance_photo_manager import CarsharingAcceptancePhotoManager
from ..order_item_requests.carsharing import CarsharingAcceptanceOrderItemRequest
from .base import BaseTariffPicker
from .carsharing_base import CarsharingBaseOrderItemManager


LOGGER = logging.getLogger(__name__)


class CarsharingAcceptanceTariffPicker(BaseTariffPicker):

    def pick_from_order_request(self, request):
        return OrderItemTariff(
            type=OrderItemTariff.Type.FIX.value,
            fix_params=FixOrderItemTariffParams(
                cost=0,
            ),
        )

    def pick_from_order_item_request(self, user, request_impl, context):
        return OrderItemTariff(
            type=OrderItemTariff.Type.FIX.value,
            fix_params=FixOrderItemTariffParams(
                cost=0,
            ),
        )


class CarsharingAcceptanceManager(CarsharingBaseOrderItemManager):

    class ActionTypes(enum.Enum):
        REPORT_CONDITION = 'report_condition'
        REPORT_PHOTO = 'report_photo'
        FINISH = 'finish'
        FINISH_FORCE = 'finish_force'

    _mds_client = import_class(cars.settings.MDS['client_class']).from_settings()

    @classmethod
    def get_item_request_class(cls):
        return CarsharingAcceptanceOrderItemRequest

    @classmethod
    def get_item_type(cls):
        return OrderItem.Type.CARSHARING_ACCEPTANCE

    @classmethod
    def _get_tariff_picker(cls):
        return CarsharingAcceptanceTariffPicker()

    @classmethod
    def materialize_request(cls, order, request, started_at=None):
        car = request.impl.get_car()
        current_acceptances = list(
            CarsharingAcceptance.objects
            .select_related('order_item')
            .filter(
                order_item__order=order,
                order_item__finished_at__isnull=True,
                car=car,
            )
        )
        if current_acceptances:
            if len(current_acceptances) != 1:
                acceptances_ids_str = ','.join([str(r.id) for r in current_acceptances])
                raise RuntimeError(
                    'multiple active acceptances per order: {}'.format(acceptances_ids_str),
                )
            return current_acceptances[0].order_item

        items_to_finish = (
            OrderItem.objects
            .filter(
                order=order,
                finished_at__isnull=True,
                type__in=[
                    OrderItem.Type.CARSHARING_PARKING.value,
                    OrderItem.Type.CARSHARING_RESERVATION.value,
                    OrderItem.Type.CARSHARING_RESERVATION_PAID.value,
                    OrderItem.Type.CARSHARING_RIDE.value,
                ],
            )
        )

        try:
            cls.try_stop_warming(car)
        except TelematicsApiResponse.Error:
            LOGGER.exception('failed to stop warming on acceptance start')

        with transaction.atomic():
            for item_to_finish in items_to_finish:
                cls._finish_item(item_to_finish, finished_at=started_at)
            item = super().materialize_request(order, request, started_at=started_at)

        return item

    def send_action(self, action, context, params=None):
        if action == self.ActionTypes.FINISH.value:
            self._send_finish_action()
        elif action == self.ActionTypes.FINISH_FORCE.value:
            self._send_finish_action(respect_telematics=False)
        elif action == self.ActionTypes.REPORT_CONDITION.value:
            self._send_report_condition_action(params)
        elif action == self.ActionTypes.REPORT_PHOTO.value:
            self._send_report_photo_action(params)
        else:
            raise self.InvalidActionError(action)

    def _send_finish_action(self, respect_telematics=True):
        with transaction.atomic():
            acceptance = self._item.get_impl()

            try:
                self.try_stop_warming(acceptance.car)
                response = self._telematics_proxy.end_lease(acceptance.car.imei)
                response.raise_for_status()
            except TelematicsApiResponse.Error as e:
                if respect_telematics:
                    code = 'car.{}'.format(e.code)
                    raise self.ActionError(code=code)

            CarUpdater(acceptance.car).update_status(Car.Status.AVAILABLE)
            self._finish_item(self._item)

    def _send_report_photo_action(self, photo):
        manager = CarsharingAcceptancePhotoManager.from_settings(
            order_item=self._item,
            mds_client=self._mds_client,
        )

        try:
            manager.add_acceptance_photo(
                id=photo['id'],
                b64content=photo['content'],
                mime_type=photo['mime_type']
            )
        except KeyError:
            raise self.ActionError(code='incomplete_params')

    def _send_report_condition_action(self, params):
        acceptance = self._item.get_impl()

        try:
            acceptance.car_condition = params['car_condition']
            acceptance.insurance_present = params['insurance_present']
            acceptance.comment = params.get('comment')
            acceptance.fuel_card_present = params['fuel_card_present']
            acceptance.sts_present = params['sts_present']
        except KeyError:
            raise self.ActionError(code='incomplete_params')

        acceptance.save()
