from django.db import transaction

from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.parking_area_checker import ParkingAreaChecker
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.car import Car
from cars.carsharing.models.parking import CarsharingParking
from ...models.order_item import OrderItem
from ..order_item_requests.carsharing import CarsharingParkingOrderItemRequest
from .base import BaseTariffPicker
from .carsharing_base import CarsharingBaseOrderItemManager


class CarsharingParkingTariffPicker(BaseTariffPicker):

    def pick_from_order_request(self, request):
        car = request.items[0].impl.get_car()
        tariff = self.calculator.get_carsharing_base_tariffs(
            user=request.user,
            cars=[car],
        )[car]
        return self._build_per_minute_tariff(cost_per_minute=tariff.parking_cost_per_minute)

    def pick_from_order_item_request(self, user, request_impl, context):
        if request_impl.terminate_for_debt:
            # Parkings to be terminated for debt are free.
            tariff = self._build_per_minute_tariff(cost_per_minute=0)
        else:
            snapshot = self._get_current_order_tariff_snapshot(user=user)
            car = request_impl.get_car()
            carsharing_base_tariff = self.calculator.get_carsharing_base_tariffs(
                user=user,
                cars=[car],
            )[car]
            current_cost_per_minute = carsharing_base_tariff.parking_cost_per_minute
            if current_cost_per_minute > 0 and snapshot is not None:
                tariff = snapshot.carsharing_reservation_paid
            else:
                tariff = self._build_per_minute_tariff(cost_per_minute=current_cost_per_minute)
        return tariff


class CarsharingParkingManager(CarsharingBaseOrderItemManager):

    _parking_area_checker = ParkingAreaChecker.from_settings()

    @classmethod
    def get_item_request_class(cls):
        return CarsharingParkingOrderItemRequest

    @classmethod
    def get_item_type(cls):
        return OrderItem.Type.CARSHARING_PARKING

    @classmethod
    def _get_tariff_picker(cls):
        return CarsharingParkingTariffPicker()

    @classmethod
    def materialize_request(cls, order, request, started_at=None):
        car = request.impl.get_car()
        current_parkings = list(
            CarsharingParking.objects
            .select_related('order_item__tariff__per_minute_params')
            .filter(
                order_item__order=order,
                order_item__finished_at__isnull=True,
                car=car,
            )
        )
        if current_parkings:
            if len(current_parkings) != 1:
                parking_ids_str = ','.join([str(r.id) for r in current_parkings])
                raise RuntimeError(
                    'multiple active parkings per order: {}'.format(parking_ids_str),
                )
            current_parking = current_parkings[0]
            if current_parking.order_item.tariff == request.tariff:
                return current_parking.order_item

        items_to_finish = (
            OrderItem.objects
            .filter(
                order=order,
                finished_at__isnull=True,
                type__in=[
                    OrderItem.Type.CARSHARING_ACCEPTANCE.value,
                    OrderItem.Type.CARSHARING_RESERVATION.value,
                    OrderItem.Type.CARSHARING_RIDE.value,
                    OrderItem.Type.CARSHARING_PARKING.value,
                ],
            )
        )

        with transaction.atomic():
            for item_to_finish in items_to_finish:
                cls._finish_item(item_to_finish, finished_at=started_at)
            item = super().materialize_request(order, request, started_at=started_at)

        return item

    def send_action(self, action, context, params=None):  # pylint: disable=unused-argument
        if action == 'finish':
            self._send_finish_action()
        elif action == 'finish_debt':
            self._send_finish_action(check_telematics=False)
        elif action == 'finish_force':
            self._send_finish_action(check_parking_area=False)
        else:
            raise self.InvalidActionError(action)

    def _send_finish_action(self, check_parking_area=True, check_telematics=True):
        with transaction.atomic():
            parking = self._item.get_impl()

            if check_parking_area:
                try:
                    self._parking_area_checker.check(location=parking.car.get_location())
                except self._parking_area_checker.Error as e:
                    raise self.ActionError(code=str(e))

            try:
                self.try_stop_warming(parking.car)
                response = self._telematics_proxy.end_lease(parking.car.imei)
                response.raise_for_status()
            except TelematicsApiResponse.Error as e:
                if check_telematics:
                    code = 'car.{}'.format(e.code)
                    raise self.ActionError(code=code)

            CarUpdater(parking.car).update_status(Car.Status.AVAILABLE)
            self._finish_item(self._item)
