from django.db import transaction

from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.car import Car
from ...models.order_item import OrderItem
from ...models.order_item_tariff import OrderItemTariff, FixOrderItemTariffParams
from ..order_item_requests.carsharing import CarsharingReservationOrderItemRequest
from .base import BaseTariffPicker
from .carsharing_base import CarsharingBaseOrderItemManager


class CarsharingReservationTariffPicker(BaseTariffPicker):

    def pick_from_order_item_request(self, user, request_impl, context):
        return OrderItemTariff(
            type=OrderItemTariff.Type.FIX.value,
            fix_params=FixOrderItemTariffParams(
                cost=0,
            ),
        )


class CarsharingReservationManager(CarsharingBaseOrderItemManager):

    @classmethod
    def get_item_request_class(cls):
        return CarsharingReservationOrderItemRequest

    @classmethod
    def get_item_type(cls):
        return OrderItem.Type.CARSHARING_RESERVATION

    @classmethod
    def _get_tariff_picker(cls):
        return CarsharingReservationTariffPicker()

    def send_action(self, action, context, params=None):  # pylint: disable=unused-argument
        if action in ('finish', 'finish_force'):
            with transaction.atomic():
                reservation = self._item.get_impl()

                try:
                    self.try_stop_warming(reservation.car)
                except TelematicsApiResponse.Error as e:
                    code = 'car.{}'.format(e.code)
                    raise self.ActionError(code=code)

                CarUpdater(reservation.car).update_status(Car.Status.AVAILABLE)
                self._finish_item(self._item)
        else:
            raise self.InvalidActionError(action)
