from django.db import transaction
from django.db.models import Q

from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.car import Car
from ...models.order_item import OrderItem
from ..order_item_requests.carsharing import CarsharingReservationPaidOrderItemRequest
from .base import BaseTariffPicker
from .carsharing_base import CarsharingBaseOrderItemManager


class CarsharingReservationPaidTariffPicker(BaseTariffPicker):

    def pick_from_order_request(self, request):
        car = request.items[0].impl.get_car()
        tariff = self.calculator.get_carsharing_base_tariffs(
            user=request.user,
            cars=[car],
        )[car]
        return self._build_per_minute_tariff(cost_per_minute=tariff.parking_cost_per_minute)

    def pick_from_order_item_request(self, user, request_impl, context):
        snapshot = self._get_current_order_tariff_snapshot(user=user)

        car = request_impl.get_car()
        carsharing_base_tariff = self.calculator.get_carsharing_base_tariffs(
            user=user,
            cars=[car],
        )[car]
        current_cost_per_minute = carsharing_base_tariff.parking_cost_per_minute

        if current_cost_per_minute > 0 and snapshot is not None:
            tariff = snapshot.carsharing_parking
        else:
            tariff = self._build_per_minute_tariff(cost_per_minute=current_cost_per_minute)

        return tariff


class CarsharingReservationPaidManager(CarsharingBaseOrderItemManager):

    @classmethod
    def get_item_request_class(cls):
        return CarsharingReservationPaidOrderItemRequest

    @classmethod
    def get_item_type(cls):
        return OrderItem.Type.CARSHARING_RESERVATION_PAID

    @classmethod
    def _get_tariff_picker(cls):
        return CarsharingReservationPaidTariffPicker()

    @classmethod
    def materialize_request(cls, order, request, started_at=None):
        car = request.impl.get_car()
        items_to_finish = (
            OrderItem.objects
            .filter(
                Q(carsharing_reservation__car=car) | Q(carsharing_reservation_paid__car=car),
                type__in=[
                    OrderItem.Type.CARSHARING_RESERVATION.value,
                    OrderItem.Type.CARSHARING_RESERVATION_PAID.value,
                ],
                order=order,
                finished_at__isnull=True,
            )
        )

        with transaction.atomic():
            for item_to_finish in items_to_finish:
                cls._finish_item(item_to_finish, finished_at=started_at)
            item = super().materialize_request(order, request, started_at=started_at)

        return item

    def send_action(self, action, context, params=None):
        if action in ('finish', 'finish_force'):
            with transaction.atomic():
                reservation = self._item.get_impl()

                try:
                    self.try_stop_warming(reservation.car)
                except TelematicsApiResponse.Error as e:
                    code = 'car.{}'.format(e.code)
                    raise self.ActionError(code=code)

                CarUpdater(reservation.car).update_status(Car.Status.AVAILABLE)
                self._finish_item(self._item)
        else:
            raise self.InvalidActionError(action)
