import logging

from django.db import transaction

from cars.core.saas_drive import SaasDrive
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.parking_area_checker import ParkingAreaChecker
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.car import Car
from cars.carsharing.models.ride import CarsharingRide
from ...iface.order_item_request import IOrderItemRequestImpl
from ...models.order_item import OrderItem
from ..order_item_requests.carsharing import CarsharingRideOrderItemRequest
from .base import BaseTariffPicker
from .carsharing_base import CarsharingBaseOrderItemManager


LOGGER = logging.getLogger(__name__)


class CarsharingRideTariffPicker(BaseTariffPicker):

    def __init__(self, *args, saas_drive_client, **kwargs):
        super().__init__(*args, **kwargs)
        self._saas_drive = saas_drive_client

    def pick_from_order_request(self, request):
        car = request.items[0].impl.get_car()
        tariff = self.calculator.get_carsharing_base_tariffs(
            user=request.user,
            cars=[car],
        )[car]
        return self._build_per_minute_tariff(cost_per_minute=tariff.ride_cost_per_minute)

    def pick_from_order_item_request(self, user, request_impl, context):
        if request_impl.fix_id is None:
            tariff = self._pick_per_minute_tariff(
                user=user,
                request_impl=request_impl,
            )
        else:
            tariff = self._pick_fix_tariff(
                user=user,
                request_impl=request_impl,
                context=context,
                fix_id=request_impl.fix_id,
            )

        return tariff

    def _pick_per_minute_tariff(self, user, request_impl):  # pylint: disable=unused-argument
        snapshot = self._get_current_order_tariff_snapshot(user=user)

        if snapshot is not None:
            tariff = snapshot.carsharing_ride
        else:
            car = request_impl.get_car()
            carsharing_base_tariff = self.calculator.get_carsharing_base_tariffs(
                user=user,
                cars=[car],
            )[car]
            tariff = self._build_per_minute_tariff(
                cost_per_minute=carsharing_base_tariff.ride_cost_per_minute,
            )

        return tariff

    def _pick_fix_tariff(self, user, request_impl, context, fix_id):  # pylint: disable=unused-argument
        try:
            offer = self._saas_drive.get_offer_info(
                oid=request_impl.fix_id,
                oauth_token=context.oauth_token,
            )
        except self._saas_drive.OfferNotFoundError:
            raise IOrderItemRequestImpl.Error('offer.not_found')

        tariff = self._build_fix_tariff(cost=offer.fix_price)

        return tariff


class CarsharingRideManager(CarsharingBaseOrderItemManager):

    _parking_area_checker = ParkingAreaChecker.from_settings()
    _saas_drive_client = SaasDrive.from_settings()

    @classmethod
    def get_item_request_class(cls):
        return CarsharingRideOrderItemRequest

    @classmethod
    def get_item_type(cls):
        return OrderItem.Type.CARSHARING_RIDE

    @classmethod
    def _get_tariff_picker(cls):
        return CarsharingRideTariffPicker(saas_drive_client=cls._saas_drive_client)

    @classmethod
    def materialize_request(cls, order, request, started_at=None):
        car = request.impl.get_car()
        current_rides = list(
            CarsharingRide.objects
            .select_related('order_item')
            .filter(
                order_item__order=order,
                order_item__finished_at__isnull=True,
                car=car,
            )
        )
        if current_rides:
            if len(current_rides) != 1:
                ride_ids_str = ','.join([str(r.id) for r in current_rides])
                raise RuntimeError('multiple active rides per order: {}'.format(ride_ids_str))
            return current_rides[0].order_item

        items_to_finish = (
            OrderItem.objects
            .filter(
                order=order,
                finished_at__isnull=True,
                type__in=[
                    OrderItem.Type.CARSHARING_ACCEPTANCE.value,
                    OrderItem.Type.CARSHARING_PARKING.value,
                    OrderItem.Type.CARSHARING_RESERVATION.value,
                    OrderItem.Type.CARSHARING_RESERVATION_PAID.value,
                ],
            )
        )

        try:
            cls.try_stop_warming(car)
        except TelematicsApiResponse.Error:
            LOGGER.exception('failed to stop warming on ride start')

        with transaction.atomic():
            for item_to_finish in items_to_finish:
                cls._finish_item(item_to_finish, finished_at=started_at)
            item = super().materialize_request(order, request, started_at=started_at)

        return item

    @classmethod
    def _finish_item_impl(cls, item, *args, **kwargs):
        ride = item.get_impl()
        ride.finish_total_mileage = ride.car.get_mileage()
        try:
            ride.save()
        except Exception:
            ride.refresh_from_db()
            raise
        super()._finish_item_impl(item, *args, **kwargs)

    def send_action(self, action, context, params=None):  # pylint: disable=unused-argument
        if action == 'finish':
            self._send_finish_action()
        elif action == 'finish_force':
            self._send_finish_action(respect_telematics=False, check_parking_area=False)
        else:
            raise self.InvalidActionError(action)

    def _send_finish_action(self, respect_telematics=True, check_parking_area=True):
        with transaction.atomic():
            ride = self._item.get_impl()

            if check_parking_area:
                try:
                    self._parking_area_checker.check(location=ride.car.get_location())
                except self._parking_area_checker.Error as e:
                    raise self.ActionError(code=str(e))

            response = self._telematics_proxy.end_lease(ride.car.imei)
            try:
                response.raise_for_status()
            except response.Error as e:
                if respect_telematics:
                    code = 'car.{}'.format(e.code)
                    raise self.ActionError(code=code)

            CarUpdater(ride.car).update_status(Car.Status.AVAILABLE)
            self._finish_item(self._item)
