import collections
import logging
import time
import uuid

from django.db import transaction
from django.utils import timezone
from django.utils.translation import gettext as _

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.parking_area_checker import ParkingAreaChecker
from cars.carsharing.models.acceptance import CarsharingAcceptance
from cars.carsharing.models.car import Car
from cars.carsharing.models.parking import CarsharingParking
from cars.carsharing.models.reservation import CarsharingReservation
from cars.carsharing.models.reservation_paid import CarsharingReservationPaid
from cars.carsharing.models.ride import CarsharingRide
from cars.core.util import import_class
from cars.users.models.user import User
from ...iface.order_item_request import IOrderItemRequestImpl  # pylint: disable=relative-beyond-top-level
from ...models.order_item import OrderItem
from ..order_debt_manager import OrderDebtManager
from ..order_operation_message import OrderOperationMessage
from ..order_operation_notifier import OrderOperationNotifier
from ..push_client import PUSH_CLIENT


LOGGER = logging.getLogger(__name__)


class SingleCarOrderItemRequestMixin(object):

    _telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def __init__(self, user, car_id, **kwargs):
        super(**kwargs)
        self._user = user
        self._car_id = car_id
        self._car = None

    @classmethod
    def from_dict(cls, user, data, **kwargs):  # pylint: disable=unused-argument
        raw_car_id = data['car_id']
        if isinstance(raw_car_id, uuid.UUID):
            car_id = raw_car_id
        else:
            try:
                car_id = uuid.UUID(data['car_id'])
            except (KeyError, ValueError):
                raise cls.ParseError('car_id.invalid')

        extra = cls._extract_extra_from_dict(data)
        kwargs.update(extra)

        return cls(
            user=user,
            car_id=car_id,
            **kwargs
        )

    @classmethod
    def _extract_extra_from_dict(cls, data):  # pylint: disable=unused-argument
        return {}

    @property
    def car_id(self):
        return self._car_id

    def get_car(self):
        if self._car is None:
            self._car = Car.objects.get(id=self._car_id)
        return self._car

    def prepare(self):
        try:
            (
                Car.objects
                .select_for_update()
                .get(id=self.get_car().id)
            )
        except Car.DoesNotExist:
            raise self.PrepareError('car.not_found')

    def _check_car_status(self, car, allowed_statuses):
        if car.get_status() not in allowed_statuses:
            LOGGER.warning(
                'unexpected car status: %s not in %s',
                car.get_status(),
                allowed_statuses,
                extra={
                    'stack': True,
                },
            )
            raise self.PrepareError('car.status.invalid')

    def _check_car_reserved(self, car):
        reservation_exists = False

        reservation_classes = [CarsharingReservation, CarsharingReservationPaid]
        for reservation_class in reservation_classes:
            reservation_object_exists = (
                reservation_class.objects
                .filter(
                    car=car,
                    order_item__order__user_id=self._user.id,
                    order_item__order__completed_at__isnull=True,
                )
            )
            if reservation_object_exists:
                reservation_exists = True
                break

        if not reservation_exists:
            raise self.PrepareError('car.not_reserved')

    def _check_car_not_reserved(self, car, allowed_user=None):
        reservation = (
            CarsharingReservation.objects
            .filter(
                car=car,
                order_item__order__completed_at__isnull=True,
            )
            .select_related('order_item__order')
            .first()
        )
        if reservation is not None:
            reservation_user_id = reservation.order_item.order.user_id
            if not (allowed_user is not None and reservation_user_id == allowed_user.id):
                raise self.PrepareError('car.already_reserved')

    def _check_acceptance_passed(self, car, user):
        acceptance = (
            CarsharingAcceptance.objects
            .filter(
                car=car,
                order_item__order__user=user,
                order_item__order__completed_at__isnull=True,
            )
            .first()
        )
        if acceptance is None:
            raise self.PrepareError('acceptance.not_started')

        if not acceptance.is_completed():
            raise self.PrepareError('acceptance.incomplete')

    def _check_user_status(self, user, allowed_statuses=None):
        if allowed_statuses is None:
            allowed_statuses = {
                User.Status.ACTIVE,
                User.Status.DEBT,
            }
        if user.get_status() not in allowed_statuses:
            raise self.PrepareError('user.inactive')
        if not user.is_phone_verified:
            raise self.PrepareError('user.phone_not_verified')

    def _safe_update_car_status_when_materializing(self, status):
        try:
            CarUpdater(self._car).update_status(status)
        except CarUpdater.BadStatusError:
            raise self.MaterializeError('car.status.invalid')


class CarsharingAcceptanceOrderItemRequest(SingleCarOrderItemRequestMixin, IOrderItemRequestImpl):

    PRE_ACCEPTANCE_STATUSES = {
        Car.Status.ACCEPTANCE,
        Car.Status.RESERVATION,
        Car.Status.RESERVATION_PAID,
    }

    def prepare(self):
        super().prepare()
        self._check_car_status(car=self._car, allowed_statuses=self.PRE_ACCEPTANCE_STATUSES)
        self._check_car_reserved(car=self._car)
        self._check_user_status(user=self._user)

    def materialize(self, order):  # pylint: disable=unused-argument
        with transaction.atomic():
            response = self._telematics_proxy.open(self._car.imei)
            try:
                response.raise_for_status()
            except response.Error as e:
                code = 'car.{}'.format(e.code)
                raise self.MaterializeError(code)

            acceptance = CarsharingAcceptance.objects.create(
                car=self._car,
            )
            self._safe_update_car_status_when_materializing(Car.Status.ACCEPTANCE)

        return acceptance


class CarsharingParkingOrderItemRequest(SingleCarOrderItemRequestMixin, IOrderItemRequestImpl):

    PRE_PARKING_STATUSES = {
        Car.Status.ACCEPTANCE,
        Car.Status.PARKING,
        Car.Status.RESERVATION,
        Car.Status.RESERVATION_PAID,
        Car.Status.RIDE,
    }

    def __init__(self, *args, terminate_for_debt=False, **kwargs):
        super().__init__(*args, **kwargs)
        self._terminate_for_debt = terminate_for_debt

    @classmethod
    def from_dict(cls, *args, user, **kwargs):
        terminator = OrderItemRequestDebtTerminator(telematics_proxy=cls._telematics_proxy)
        kwargs['terminate_for_debt'] = terminator.check_terminate_for_debt(user)
        return super().from_dict(*args, user=user, **kwargs)

    @property
    def terminate_for_debt(self):
        return self._terminate_for_debt

    def prepare(self):
        super().prepare()
        self._check_car_status(car=self._car, allowed_statuses=self.PRE_PARKING_STATUSES)
        self._check_car_reserved(car=self._car)
        self._check_user_status(user=self._user)

    def materialize(self, order):  # pylint: disable=unused-argument
        with transaction.atomic():
            response = self._telematics_proxy.end_lease(self._car.imei)
            try:
                response.raise_for_status()
            except response.Error as e:
                code = 'car.{}'.format(e.code)
                raise self.MaterializeError(code)

            debt_termination_max_duration_seconds = None
            if self.terminate_for_debt:
                terminator = OrderItemRequestDebtTerminator(
                    telematics_proxy=self._telematics_proxy,
                )
                debt_termination_max_duration_seconds = terminator.try_terminate_for_debt(
                    order=order,
                    car=self._car,
                )

            parking = CarsharingParking.objects.create(
                car=self._car,
                debt_termination_max_duration_seconds=debt_termination_max_duration_seconds,
            )
            self._safe_update_car_status_when_materializing(Car.Status.PARKING)

        return parking


class CarsharingReservationOrderItemRequest(SingleCarOrderItemRequestMixin, IOrderItemRequestImpl):

    PRE_RESERVATION_STATUSES = {
        Car.Status.AVAILABLE,
        Car.Status.RESERVATION,
        Car.Status.RESERVATION_PAID,
    }

    def __init__(self, *args, max_duration_seconds, message=None, **kwargs):
        super().__init__(*args, **kwargs)
        self._max_duration_seconds = max_duration_seconds
        self._message = message

    @classmethod
    def from_dict(cls, user, data):
        max_duration_seconds, message = cls._pick_max_duration_seconds(user)
        return super().from_dict(
            user=user,
            data=data,
            max_duration_seconds=max_duration_seconds,
            message=message,
        )

    @classmethod
    def _pick_max_duration_seconds(cls, user):
        settings = cars.settings.CARSHARING['reservation']

        now = timezone.now()
        is_constrained = False

        for constraint in settings['constraints']:
            span_start = now - constraint['span']

            items = (
                OrderItem.objects
                .filter(
                    order__user=user,
                    started_at__gte=span_start,
                )
            )

            item_types_per_order = collections.defaultdict(set)
            for item in items:
                item_types_per_order[item.order_id].add(item.get_type())

            n_orders_without_rides = len([
                order_id for order_id, item_types in item_types_per_order.items()
                if OrderItem.Type.CARSHARING_RIDE not in item_types
            ])

            if n_orders_without_rides >= constraint['count']:
                is_constrained = True
                break

        message = None
        if is_constrained:
            max_duration_seconds = settings['constrained_max_duration_seconds']
            message = OrderOperationMessage(
                title=_('orders.order_items.carsharing_reservation.no_free_period.title'),
                body=_('orders.order_items.carsharing_reservation.no_free_period.body'),
                action=_('orders.order_items.carsharing_reservation.no_free_period.action'),
            )
        else:
            LOGGER.info('user %s did not abuse free reservation', str(user.id))
            max_duration_seconds = settings['full_max_duration_seconds']

            # Add bonus waiting time minutes for Yandex.Plus users
            plus_settings = cars.settings.CARSHARING['plus']
            if user.get_plus_status():
                max_duration_seconds = plus_settings['reservation_full_max_duration_seconds']

        return max_duration_seconds, message

    def get_message(self):
        return self._message

    def prepare(self):
        super().prepare()
        self._check_car_status(car=self._car, allowed_statuses=self.PRE_RESERVATION_STATUSES)
        self._check_car_not_reserved(car=self._car, allowed_user=self._user)
        self._check_user_status(user=self._user, allowed_statuses=[User.Status.ACTIVE])

    def materialize(self, order):  # pylint: disable=unused-argument
        with transaction.atomic():
            car_location = self._car.get_location()
            reservation = CarsharingReservation.objects.create(
                car=self._car,
                max_duration_seconds=self._max_duration_seconds,
                car_location_course=car_location.course if car_location else None,
                car_location_lat=car_location.lat if car_location else None,
                car_location_lon=car_location.lon if car_location else None,
            )
            self._safe_update_car_status_when_materializing(Car.Status.RESERVATION)
        return reservation


class CarsharingReservationPaidOrderItemRequest(SingleCarOrderItemRequestMixin,
                                                IOrderItemRequestImpl):

    PRE_RESERVATION_STATUSES = {
        Car.Status.RESERVATION,
        Car.Status.RESERVATION_PAID,
    }

    def prepare(self):
        super().prepare()
        self._check_car_status(car=self._car, allowed_statuses=self.PRE_RESERVATION_STATUSES)
        self._check_car_reserved(car=self._car)
        self._check_user_status(user=self._user)

    def materialize(self, order):  # pylint: disable=unused-argument
        with transaction.atomic():
            car_location = self._car.get_location()
            reservation = CarsharingReservationPaid.objects.create(
                car=self._car,
                car_location_course=car_location.course if car_location else None,
                car_location_lat=car_location.lat if car_location else None,
                car_location_lon=car_location.lon if car_location else None,
            )
            self._safe_update_car_status_when_materializing(Car.Status.RESERVATION_PAID)

        return reservation


class CarsharingRideOrderItemRequest(SingleCarOrderItemRequestMixin, IOrderItemRequestImpl):

    PRE_RIDE_STATUSES = {
        Car.Status.ACCEPTANCE,
        Car.Status.PARKING,
        Car.Status.RESERVATION,
        Car.Status.RIDE,
    }

    def __init__(self, *args, fix_id, **kwargs):
        super().__init__(*args, **kwargs)
        self._fix_id = fix_id

    @classmethod
    def _extract_extra_from_dict(cls, data):
        return {
            'fix_id': data.get('fix_id'),
        }

    @property
    def fix_id(self):
        return self._fix_id

    def prepare(self):
        super().prepare()
        self._check_car_status(car=self._car, allowed_statuses=self.PRE_RIDE_STATUSES)
        self._check_car_reserved(car=self._car)
        self._check_user_status(user=self._user)
        self._check_acceptance_passed(car=self._car, user=self._user)

    def materialize(self, order):  # pylint: disable=unused-argument
        with transaction.atomic():
            ride = CarsharingRide.objects.create(
                car=self._car,
                fix_id=self._fix_id,
                start_total_mileage=self._car.get_mileage(),
                finish_total_mileage=None
            )
            self._safe_update_car_status_when_materializing(Car.Status.RIDE)

            response = self._telematics_proxy.start_lease(self._car.imei)
            try:
                response.raise_for_status()
            except response.Error as e:
                code = 'car.{}'.format(e.code)
                raise self.MaterializeError(code)

        return ride


class OrderItemRequestDebtTerminator:

    _order_debt_manager = OrderDebtManager.from_settings(push_client=PUSH_CLIENT)
    _order_operation_notifier = OrderOperationNotifier.from_settings()
    _parking_area_checker = ParkingAreaChecker.from_settings()

    def __init__(self, telematics_proxy):
        self._telematics_proxy = telematics_proxy

    def check_terminate_for_debt(self, user):
        terminate_for_debt = self._order_debt_manager.check_if_terminate_order_for_debt(user)
        return terminate_for_debt

    def try_terminate_for_debt(self, order, car):
        is_in_zone = True
        try:
            self._parking_area_checker.check(location=car.get_location())
        except self._parking_area_checker.Error:
            is_in_zone = False

        debt_termination_max_duration_seconds = None

        if is_in_zone:
            debt_termination_max_duration_seconds = (
                self._order_debt_manager.debt_order_termination_delay
            )
            self._prepare_for_debt_termination(
                order=order,
                car=car,
                debt=self._order_debt_manager.get_debt(order.user),
            )

        return debt_termination_max_duration_seconds

    def _prepare_for_debt_termination(self, order, car, debt):
        time.sleep(self._telematics_proxy.POST_END_LEASE_DELAY)

        response = self._telematics_proxy.open(car.imei)
        try:
            response.raise_for_status()
        except response.Error as e:
            code = 'car.{}'.format(e.code)
            raise IOrderItemRequestImpl.MaterializeError(code)

        try:
            self._order_operation_notifier.notify_order_terminating_for_debt(
                user=order.user,
                debt=debt,
            )
        except Exception:
            LOGGER.exception('failed to notify of terminating order for debt')
