import collections
import logging

from django.db import connections, transaction
from django.utils import timezone

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.models import Car
from cars.core.trust import TrustClient
from cars.users.core.user_profile_updater import UserProfileUpdater
from cars.users.models.app_install import AppInstall
from cars.users.models.user import User
from ..models.order import Order
from ..models.order_item import OrderItem
from ..models.order_payment_method import OrderPaymentMethod
from ..models.order_tariff_snapshot import OrderTariffSnapshot
from ..serializers.order import OrderSerializer
from .order_debt_manager import OrderDebtManager
from .order_item_managers.base import OrderItemActionContext
from .order_item_managers.carsharing_parking import CarsharingParkingManager
from .order_item_managers.carsharing_reservation_paid import CarsharingReservationPaidManager
from .order_item_managers.carsharing_ride import CarsharingRideManager
from .order_item_managers.factory import OrderItemManagerFactory
from .order_operation_notifier import OrderOperationNotifier
from .order_payment_processor import OrderPaymentProcessor
from .preliminary_payments_manager import OrderPreliminaryPaymentsManager


LOGGER = logging.getLogger(__name__)


class OrderManager(object):

    class Error(Exception):
        pass

    class OrderNotFoundError(Error):
        pass

    def __init__(self, trust_client, order_payment_processor, order_debt_manager,
                 preliminary_payments_manager, order_operation_notifier, push_client):
        self._trust = trust_client
        self._order_payment_processor = order_payment_processor
        self._order_debt_manager = order_debt_manager
        self._preliminary_payments_manager = preliminary_payments_manager
        self._notifier = order_operation_notifier
        self._push_client = push_client

    @classmethod
    def from_settings(cls, push_client):
        trust_client = TrustClient.from_settings(push_client=push_client)
        return cls(
            trust_client=trust_client,
            order_payment_processor=OrderPaymentProcessor.from_settings(),
            order_debt_manager=OrderDebtManager.from_settings(push_client=push_client),
            preliminary_payments_manager=OrderPreliminaryPaymentsManager.from_settings(),
            order_operation_notifier=OrderOperationNotifier.from_settings(),
            push_client=push_client,
        )

    def _check_app(self, user):
        try:
            app_install = AppInstall.objects.get(user=user, is_latest=True)
        except AppInstall.DoesNotExist:
            raise self.Error('app.version.unknown')

        platform = app_install.get_platform()

        # Remove debt-specific check after debtors update.
        from cars.core.constants import AppPlatform
        if user.get_status() is User.Status.DEBT:
            min_app_version = {
                AppPlatform.ANDROID: '1.0.7',
                AppPlatform.IOS: '113.0.0',
            }[platform]
            min_app_build = {
                AppPlatform.ANDROID: '180',
                AppPlatform.IOS: '370',
            }[platform]
        else:
            min_app_version = cars.settings.DRIVE['min_app_version'][platform]
            min_app_build = cars.settings.DRIVE['min_app_build'][platform]

        if app_install.app_version_less_than(min_app_version):
            raise self.Error('app.version.deprecated')

        if app_install.app_build_less_than(min_app_build):
            raise self.Error('app.build.deprecated')

    def create_order(self, order_request):
        if order_request.user.get_new_backend_migration_shard() <= cars.settings.MAX_DISABLED_SHARD:
            raise self.Error('app.version.deprecated')

        user_id = str(order_request.user.id)
        if self._has_new_backend_order(user_id):
            raise self.Error('order.misc_error')

        result = self._materialize_order_request(order_request)
        self._log(subtype='created', order=result.order)

        car_manufacturer = result.order.get_sorted_items()[0].carsharing_reservation.car.model.manufacturer

        is_porsche = car_manufacturer == 'Porsche'

        try:
            if not is_porsche:
                self._preliminary_payments_manager.maybe_make_preliminary_payment(order=result.order)
        except self._preliminary_payments_manager.PreliminaryPaymentError:
            LOGGER.info('cancelling order due to preliminary payment error: %s', result.order.id)
            with transaction.atomic():
                self.force_complete_order(result.order)
                self._order_payment_processor.finalize_order_payment_status(result.order)
                self._order_payment_processor.refund_order(result.order)
            raise self.Error('preliminary_payment.error')

        return result

    def create_service_app_order(self, order_request):
        """
        Service app scenarios doesn't require premilinary payments.
        Don't check for them and don't throw errors in case preauthorizarion is failed.
        """
        result = self._materialize_order_request(order_request)
        self._log(subtype='created', order=result.order)

        return result

    def _has_new_backend_order(self, user_id):
        query = """
        select * from billing_tasks
        where user_id='{}' and billing_type='car_usage';
        """.format(user_id)

        has_order = False
        cursor = connections[cars.settings.DB_RO_ID].cursor()
        try:
            cursor.execute(query)
            result = cursor.fetchall()
            for _ in result:
                has_order = True
        except Exception:
            LOGGER.exception('unable to fetch billing tasks from new backend for user %s', user_id)

        return has_order

    def _materialize_order_request(self, order_request):
        with transaction.atomic():
            # Write lock the user to avoid concurrent execution.
            user = (
                User.objects
                .select_for_update()
                .get(id=order_request.user.id)
            )

            try:
                self._check_app(user=user)
            except self.Error:
                raise
            except Exception:
                # Don't block order creation by a failed check.
                LOGGER.exception('failed to check app version for user %s', user.id)

            # A concurrent operation may have already created the order.
            order = (
                Order.objects
                .filter(
                    user=order_request.user,
                    completed_at__isnull=True,
                )
                .first()
            )
            if order:
                LOGGER.warning(
                    'user %s already has an active order %s',
                    order_request.user.id,
                    order.id,
                )
                return OrderOperationResult(
                    order=order,
                    message=None,
                )

            order_request.prepare()

            payment_method = self._create_payment_method(user=user, request=order_request)

            order = Order.objects.create(
                user=order_request.user,
                payment_method=payment_method,
                created_at=timezone.now(),
                has_plus_discount=order_request.user.get_plus_status(),
            )

            self._create_tariff_snapshot(request=order_request, order=order)

            order_request.materialize(order)

        return OrderOperationResult(
            order=order,
            message=order_request.message,
        )

    def _create_payment_method(self, user, request):
        credit_card = user.get_credit_card()

        if credit_card:
            if credit_card.paymethod_id is None:
                updater = UserProfileUpdater(user, trust_client=self._trust)
                try:
                    credit_card = updater.update_credit_card(
                        pan_prefix=credit_card.pan_prefix,
                        pan_suffix=credit_card.pan_suffix,
                    )
                except updater.CreditCardNotBound:
                    LOGGER.warning('credit card for user %s not bound', user.id)
                    credit_card = None

            if credit_card is None or credit_card.paymethod_id is None:
                payment_method = None
            else:
                payment_method = OrderPaymentMethod.objects.create(
                    type=OrderPaymentMethod.Type.CARD.value,
                    card_paymethod_id=credit_card.paymethod_id,
                )

        elif request.payment_method:
            payment_method = OrderPaymentMethod.objects.create(
                type=request.payment_method.type.value,
                card_paymethod_id=request.payment_method.card_paymethod_id,
            )

        else:
            payment_method = None

        return payment_method

    def _create_tariff_snapshot(self, request, order):
        with transaction.atomic():
            carsharing_reservation_paid = (
                CarsharingReservationPaidManager.pick_from_order_request(request)
            )
            carsharing_ride = CarsharingRideManager.pick_from_order_request(request)
            carsharing_parking = CarsharingParkingManager.pick_from_order_request(request)

            if carsharing_reservation_paid:
                carsharing_reservation_paid.save()
            if carsharing_ride:
                carsharing_ride.save()
            if carsharing_parking:
                carsharing_parking.save()

            snapshot = OrderTariffSnapshot.objects.create(
                order=order,
                carsharing_reservation_paid=carsharing_reservation_paid,
                carsharing_ride=carsharing_ride,
                carsharing_parking=carsharing_parking,
            )

        return snapshot

    def add_order_item(self, order, order_item_request, started_at=None):
        LOGGER.info('adding %s to order %s', order_item_request.item_type, order.id)

        with transaction.atomic():
            order = Order.objects.select_for_update().get(id=order.id)
            assert order.completed_at is None

            manager_class = OrderItemManagerFactory.get_class_from_item_type(
                order_item_request.item_type,
            )
            manager_class.prepare_request(request=order_item_request)
            manager_class.materialize_request(
                order=order,
                request=order_item_request,
                started_at=started_at,
            )

        result = OrderOperationResult(order=order, message=None)

        return result

    def get_active_order_item(self, user, order_item_types, with_related=False):
        if not isinstance(order_item_types, list):
            order_item_types = [order_item_types]

        raw_order_item_types = [t.value for t in order_item_types]

        qs = OrderItem.objects
        if with_related:
            qs = qs.with_related()

        order_item = (
            qs
            .filter(
                order__user=user,
                type__in=raw_order_item_types,
                finished_at__isnull=True,
            )
            .first()
        )

        return order_item

    def send_action(self, order_item, action, context, params=None):
        with transaction.atomic():
            order = Order.objects.select_for_update().get(id=order_item.order_id)
            if order.completed_at is not None:
                assert order_item.finished_at is not None
                LOGGER.warning(
                    'trying to send action to item of a completed order %s',
                    order_item.id,
                )
                return order

            if order_item.finished_at is not None:
                LOGGER.warning(
                    'trying to send action to a finished item %s',
                    order_item.id,
                )
                return order

            item_manager = OrderItemManagerFactory.build_from_item(order_item)
            item_manager.send_action(action=action, context=context, params=params)
            self.maybe_complete_order(order)

            self._log(
                subtype='item.action',
                order=order,
                extra={
                    'action': action,
                    'params': params,
                    'order_item': {
                        'id': str(order_item.id),
                    },
                },
            )

        return order

    def maybe_complete_order(self, order):
        """Complete the order if all of its items are finished."""

        all_items_are_finished = True
        for item in order.items.all():
            if item.finished_at is None:
                all_items_are_finished = False
                break

        if all_items_are_finished:
            order = self._do_complete_order(order)

        return order

    def force_complete_order(self, order):
        order_items = OrderItem.objects.filter(order=order)
        context = OrderItemActionContext()

        with transaction.atomic():
            for order_item in order_items:
                if order_item.finished_at is not None:
                    continue
                self.send_action(order_item, 'finish_force', context=context)
            order = self._do_complete_order(order)

        return order

    def force_complete_order_ignore_telematics(self, order):
        """
        That's usually a bad idea.

        The main use case is when there's an active order for a car with big lag in
        telematics. In this case, regular finishing won't work, since it checks
        doors' (and not only doors') statuses with telematics, which won't respond.

        On the other hand, since telematics is not checked, you may end up with opened
        car and unlocked engine in AVAILABLE status. Use with extreme caution and only
        if you sure that you know what you're doing.
        """
        order_items = OrderItem.objects.filter(order=order)

        with transaction.atomic():
            for order_item in order_items:
                if order_item.finished_at is None:
                    order_item.finished_at = timezone.now()
                    order_item.save()
            order.completed_at = timezone.now()
            order.save()

            car = order.get_sorted_items()[0].get_impl().car
            car.status = 'available'
            car.save()

        return order

    def _do_complete_order(self, order):
        assert order.completed_at is None, 'order {} has been already completed'.format(order.id)

        order.completed_at = timezone.now()
        order.save()

        try:
            self._order_payment_processor.make_payments(order)
        except self._order_payment_processor.PaymentMethodMissingError:
            LOGGER.exception('missing payment method for order %s', order.id)
        except Exception:
            LOGGER.exception('failed to make payments after order: %s', order.id)

        self._log(subtype='completed', order=order)

        return order

    def _log(self, subtype, order, extra=None):
        try:
            full_type = 'order.{}'.format(subtype)

            data = {
                'order': OrderSerializer(order).data,
            }
            if extra:
                data.update(extra)

            self._push_client.log(type_=full_type, data=data)
        except Exception:
            LOGGER.exception('failed to log order state')


OrderOperationResult = collections.namedtuple('OrderOperationResult', ['order', 'message'])
