import collections
import logging

from django.db import transaction
from django.utils import timezone

from ..iface.order_item_request import IOrderItemRequestImpl
from ..models.order_item import OrderItem
from ..models.order_payment_method import OrderPaymentMethod
from .order_item_managers.factory import OrderItemManagerFactory


LOGGER = logging.getLogger(__name__)


class OrderRequest(object):

    class Error(Exception):
        pass

    def __init__(self, user, payment_method, items, created_at=None):
        if not items:
            raise self.Error('items.empty')

        if created_at is None:
            created_at = timezone.now()

        self._user = user
        self._payment_method = payment_method
        self._items = items
        self._created_at = created_at

    @property
    def user(self):
        return self._user

    @property
    def payment_method(self):
        return self._payment_method

    @property
    def items(self):
        return self._items

    @property
    def created_at(self):
        return self._created_at

    @property
    def message(self):
        message = None

        messages = [item.message for item in self._items if item.message is not None]
        if messages:
            if len(messages) > 1:
                LOGGER.warning('multiple messages for order request: %s', messages)
            message = messages[0]

        return message

    @classmethod
    def from_dict(cls, user, data, context):
        items = []

        if data.get('payment_method') is not None:
            payment_method = OrderPaymentMethodRequest.from_dict(data['payment_method'])
        else:
            payment_method = None

        for raw_item in data['items']:
            item = OrderItemRequest.from_dict(user=user, data=raw_item, context=context)
            items.append(item)

        obj = cls(
            user=user,
            payment_method=payment_method,
            items=items,
        )

        return obj

    def prepare(self):
        for item_request in self._items:
            item_request.prepare()

    def materialize(self, order):
        for item_request in self._items:
            item_request.materialize(order=order)


class OrderPaymentMethodRequest(object):

    class Error(Exception):
        pass

    def __init__(self, type_, card_paymethod_id):
        self._type = type_

        if self._type is OrderPaymentMethod.Type.CARD:
            if card_paymethod_id is None:
                raise self.Error('payment_method.params.invalid')

        self._card_paymethod_id = card_paymethod_id

    @classmethod
    def from_dict(cls, data):
        return cls(
            type_=OrderPaymentMethod.Type(data['type']),
            card_paymethod_id=data.get('card_paymethod_id'),
        )

    @property
    def type(self):
        return self._type

    @property
    def card_paymethod_id(self):
        return self._card_paymethod_id


class OrderItemRequest(object):

    class Error(Exception):
        pass

    class ParseError(Error):
        pass

    def __init__(self, user, item_type, impl, tariff):
        self._user = user
        self._item_type = item_type
        self._impl = impl
        self._tariff = tariff

    @property
    def item_type(self):
        return self._item_type

    @property
    def impl(self):
        return self._impl

    @property
    def message(self):
        return self._impl.get_message()

    @property
    def tariff(self):
        return self._tariff

    @classmethod
    def from_dict(cls, user, data, context):
        impl = None
        impl_raw_type = OrderItem.Type(data['type'])
        impl_params = data['params']

        manager_class = OrderItemManagerFactory.get_class_from_item_type(impl_raw_type)
        impl_class = manager_class.get_item_request_class()
        try:
            impl = impl_class.from_dict(
                user=user,
                data=impl_params,
            )
        except IOrderItemRequestImpl.ParseError as e:
            code = '{}.{}'.format(impl_raw_type.value, str(e))
            raise cls.ParseError(code)

        tariff = manager_class.pick_from_order_item_request(
            user=user,
            request_impl=impl,
            context=context,
        )

        return cls(
            user=user,
            item_type=manager_class.get_item_type(),
            impl=impl,
            tariff=tariff,
        )

    def prepare(self):
        try:
            self._impl.prepare()
        except IOrderItemRequestImpl.PrepareError as e:
            code = '{}.{}'.format(self._item_type.value, str(e))
            raise self.Error(code)

    def materialize(self, order, started_at=None):
        assert self._user == order.user

        if started_at is None:
            started_at = timezone.now()

        with transaction.atomic():
            try:
                item_value = self._impl.materialize(order)
            except IOrderItemRequestImpl.MaterializeError as e:
                code = '{}.{}'.format(self._item_type.value, str(e))
                raise self.Error(code)

            if self._tariff and self._tariff.id is None:
                self._tariff.save()

            item = OrderItem.objects.create(
                order=order,
                type=self._item_type.value,
                started_at=started_at,
                tariff=self._tariff,
                **{self._item_type.value: item_value}
            )

        return item


OrderItemRequestContext = collections.namedtuple(
    'OrderItemRequestContext',
    [
        'oauth_token',
    ],
)
