import decimal
import logging

import cars.settings
from cars.billing.iface.payment import IPayment
from ..models.order import Order
from ..models.order_item_payment import OrderItemPayment
from .order_payment_processor import OrderItemPaymentSpec, OrderPaymentProcessor, OrderPaymentSpec
from .user_tags import OrdersUserTags


LOGGER = logging.getLogger(__name__)


class OrderPreliminaryPaymentsManager:

    class PreliminaryPaymentError(Exception):
        pass

    def __init__(self, order_payment_processor, amount, n_orders_threshold,
                 enable_fraction_above_threshold, enable_fraction_below_threshold):
        self._order_payment_processor = order_payment_processor
        self._amount = amount
        self._n_orders_treshold = n_orders_threshold
        self._enable_fraction_above_threshold = enable_fraction_above_threshold
        self._enable_fraction_below_threshold = enable_fraction_below_threshold

    @classmethod
    def from_settings(cls):
        settings = cars.settings.ORDERS['preliminary_payments']
        return cls(
            order_payment_processor=OrderPaymentProcessor.from_settings(),
            amount=settings['amount'],
            n_orders_threshold=settings['n_orders_threshold'],
            enable_fraction_above_threshold=settings['enable_fraction_above_threshold'],
            enable_fraction_below_threshold=settings['enable_fraction_below_threshold'],
        )

    def get_default_amount(self, user):
        if user.has_tag(OrdersUserTags.NO_PRELIMINARY_PAYMENTS.value):
            return decimal.Decimal(0)

        if user.has_tag(OrdersUserTags.FORCE_PRELIMINARY_PAYMENTS.value):
            return self._amount

        orders = list(
            Order.objects
            .with_payments()
            .filter(
                user=user,
                items__carsharing_ride__isnull=False,
            )
            .distinct()
            .order_by('-created_at')
            [:self._n_orders_treshold]
        )

        error_orders = []
        for order in orders:
            has_error = False
            for item in order.get_sorted_items():
                if has_error:
                    break
                for payment in item.payments.all():
                    if payment.get_impl().is_error():
                        has_error = True
                        error_orders.append(order)
                        break

        if len(orders) < self._n_orders_treshold:
            threshold = self._enable_fraction_below_threshold
        else:
            threshold = self._enable_fraction_above_threshold

        pp_enabled = not(orders) or len(error_orders) / len(orders) >= threshold
        amount = self._amount if pp_enabled else decimal.Decimal(0)

        return amount

    def maybe_make_preliminary_payment(self, order):
        amount = self.get_default_amount(user=order.user)
        if amount <= 0:
            return

        if OrderItemPayment.objects.filter(order_item__order=order).exists():
            LOGGER.warning(
                'trying to create a preliminary payment for already paid order: %s',
                order.id,
            )
            return

        spec = OrderPaymentSpec(
            order=order,
            payment_method=order.payment_method,
            item_specs=[
                OrderItemPaymentSpec(
                    order_item=order.get_sorted_items()[0],
                    amount=0,
                ),
            ],
            allowed_payment_methods=[OrderItemPayment.PaymentMethod.CARD],
            payment_amount=amount,
        )
        self._order_payment_processor.make_payments_by_spec(spec)

        payment = OrderItemPayment.objects.get(order_item__order=order)
        self._order_payment_processor.wait_for_payments(
            payments=[payment],
            timeout=22,
        )

        if payment.get_impl().get_generic_status() is IPayment.GenericStatus.ERROR:
            raise self.PreliminaryPaymentError
