import logging
import datetime
from decimal import Decimal

import pytz
from django.utils import timezone

from kubiki.util import make_requests_session

import cars.settings
from cars.orders.models import OrderItem
from cars.carsharing.models import CarsharingRideInsuranceReport
from cars.users.core.datasync import DataSyncDocumentsClient


LOGGER = logging.getLogger(__name__)


class RenaissanceClient:

    def __init__(self, endpoint_url, datasync_client, requests_session):
        self._endpoint_url = endpoint_url
        self._datasync = datasync_client
        self._session = requests_session

    @classmethod
    def from_settings(cls):
        settings = cars.settings.THIRD_PARTY_APIS['renaissance']
        return cls(
            endpoint_url=settings['endpoint_url'],
            datasync_client=DataSyncDocumentsClient.from_settings(),
            requests_session=make_requests_session(),
        )

    def report_new_rides(self):
        unprocessed_order_items = (
            OrderItem.objects
            .filter(
                type=OrderItem.Type.CARSHARING_RIDE.value,
                finished_at__isnull=False,
                carsharing_ride__insurance_report__isnull=True,
            )
            .select_related(
                'carsharing_ride__car__insurance',
                'order__user'
            )
        )

        num_reported = 0

        for order_item in unprocessed_order_items:
            LOGGER.info('Reporting about order item with id=%s', str(order_item.id))
            try:
                self.submit_order_item(order_item)
                num_reported += 1
            except Exception:
                LOGGER.exception('Unable to report order item with id=%s', str(order_item.id))

        return num_reported

    def submit_order_item(self, ride_order_item):
        # Payload format is according to https://st.yandex-team.ru/CARSHARING-85
        user_info = self._datasync.get_passport_unverified(ride_order_item.order.user.uid, ride_order_item.order.user.passport_ds_revision) or {}

        carsharing_ride = ride_order_item.get_impl()
        car = carsharing_ride.car
        car_insurance = car.insurance

        if car_insurance.agreement_number is None:
            LOGGER.error('No insurance agreement number for car %s', str(car.id))
            return

        insurance_cost = self._calculate_ride_cost(
            car_insurance=car_insurance,
            ride_start=ride_order_item.started_at,
            ride_finish=ride_order_item.finished_at,
        )

        payload = {
            # Ya.Carsharing identification
            'Key': str(ride_order_item.id).upper(),
            'AgreementNum': car_insurance.agreement_number,
            'AgreementPartnerNum': car_insurance.agreement_partner_number,

            # Ride info
            'StartTime': ride_order_item.started_at.isoformat(),
            'EndTime': ride_order_item.finished_at.isoformat(),
            'Cost': '{0:.2f}'.format(float(insurance_cost)),
        }

        # Add user info, if present

        if user_info:
            # All of these fields are required in verified DS, so
            # if one of them is present, then the rest is present too
            payload.update(
                {
                    'DriverSurname': user_info['last_name'],
                    'DriverFirstname': user_info['first_name'],
                    'DriverPatronymic': user_info['middle_name'],
                }
            )

            date_of_birth_dt = datetime.datetime.strptime(
                user_info['birth_date'],
                '%Y-%m-%dT%H:%M:%S.%fZ',
            )
            date_of_birth_dt = timezone.make_aware(date_of_birth_dt, timezone=pytz.UTC)
            payload['DateOfBirth'] = date_of_birth_dt.isoformat()

        if car.vin is not None:
            payload['VIN'] = car.vin
        else:
            LOGGER.error('VIN is not specified for car with id=%s', car.id)

        r = self._session.post(
            url=self._endpoint_url,
            json=payload,
            timeout=60,
        )
        r.raise_for_status()

        is_succeeded = r.json()['Result']
        assert is_succeeded

        insurance_record = CarsharingRideInsuranceReport(
            carsharing_ride=carsharing_ride,
            reported_at=timezone.now(),
            start_time=ride_order_item.started_at,
            finish_time=ride_order_item.finished_at,
            cost=insurance_cost,
        )
        insurance_record.save()

        LOGGER.info('Reported successfully!')

    def _calculate_ride_cost(self, car_insurance, ride_start, ride_finish):
        minutes_duration = Decimal((ride_finish - ride_start).total_seconds() / 60.0)
        return (
            car_insurance.base_cost + minutes_duration * car_insurance.per_minute_cost
        )
