import datetime

from django.db.models import Sum
from django.utils import timezone

from cars.carsharing.models import CarsharingRideInsuranceReport
from cars.core.daemons import CarsharingDaemon
from ..models import OrderItem


class OrdersMonitoringDaemon(CarsharingDaemon):

    tick_interval = '* * * * * */15'

    def get_distributed_lock_relative_path(self):
        return 'orders/locks/monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'orders.monitoring'

    def get_solomon_service(self):
        return 'orders'

    def _monitor_not_reported_rides(self):
        """
        How many ride segments had been already completed, but not yet reported to Renaissance.
        """
        not_reported_rides = (
            OrderItem.objects
            .filter(
                type=OrderItem.Type.CARSHARING_RIDE.value,
                finished_at__isnull=False,
                carsharing_ride__insurance_report__isnull=True,
            )
            .count()
        )
        self.solomon.set_value(
            'orders.insurance.not_reported',
            not_reported_rides,
        )

    def _monitor_reported_rides(self):
        """
        How many ride segments had already been reported to Renaissance.
        """
        reported_rides = (
            CarsharingRideInsuranceReport.objects
            .filter(reported_at__gte=timezone.now() - datetime.timedelta(days=1))
            .count()
        )
        self.solomon.set_value(
            'orders.insurance.reported',
            reported_rides,
        )

    def _monitor_total_money_spent_on_insurance(self):
        """
        How much money had been spent on insurance.
        """
        total_money_spent = (
            CarsharingRideInsuranceReport.objects
            .filter(reported_at__gte=timezone.now() - datetime.timedelta(days=1))
            .aggregate(Sum('cost'))['cost__sum'] or 0
        )
        self.solomon.set_value(
            'orders.insurance.money_spent',
            float(total_money_spent),
        )

    def _monitor_rides_to_insurance_delivery(self):
        self._monitor_not_reported_rides()
        self._monitor_reported_rides()
        self._monitor_total_money_spent_on_insurance()

    def _do_tick(self):
        self._monitor_rides_to_insurance_delivery()
