import enum
import operator
import uuid

from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.db.models import Prefetch, prefetch_related_objects

from cars.users.models.user import User
from .order_payment_method import OrderPaymentMethod


class OrderManager(models.Manager):

    def with_related(self):
        from .order_item import OrderItem
        return (
            self.get_queryset()
            .select_related('payment_method', 'user')
            .prefetch_related(
                Prefetch(
                    'items',
                    queryset=OrderItem.objects.with_related(),
                )
            )
        )

    def with_payments(self):
        from .order_item import OrderItem
        return (
            self.get_queryset()
            .select_related('payment_method', 'user')
            .prefetch_related(Prefetch('items', queryset=OrderItem.objects.with_payments()))
        )


class Order(models.Model):

    class PaymentStatus(enum.Enum):
        NEW = 'new'
        SUCCESS = 'success'
        ERROR = 'error'
        REFUNDED = 'refunded'
        CANCELLED = 'cancelled'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    user = models.ForeignKey(
        User,
        on_delete=models.CASCADE,
        related_name='orders',
        db_index=False,
    )
    payment_method = models.OneToOneField(
        OrderPaymentMethod,
        on_delete=models.CASCADE,
        related_name='order',
        db_index=False,
        null=True,
    )
    payment_status = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in PaymentStatus],
        default=PaymentStatus.NEW.value,
    )

    created_at = models.DateTimeField()
    completed_at = models.DateTimeField(null=True)

    has_plus_discount = models.NullBooleanField()

    objects = OrderManager()

    class Meta:
        db_table = 'order'
        indexes = [
            models.Index(
                fields=['user'],
                name='order_user_idx',
            ),
            models.Index(
                fields=['payment_method'],
                name='order_payment_method_idx',
            ),
            models.Index(
                fields=['payment_status'],
                name='order_payment_status_idx',
            ),
        ]

    def __repr__(self):
        return '<Order: created_at={}, completed_at={}>'.format(self.created_at, self.completed_at)

    def prefetch_related_objects(self):
        prefetch_related_objects(
            [self],
            Prefetch('items', queryset=self.items.with_related()),
        )
        return self

    def get_payment_status(self):
        return self.PaymentStatus(self.payment_status)

    def get_sorted_items(self):
        # Use .all() instead of .order_by('...') for prefetch_related to work.
        return sorted(self.items.all(), key=operator.attrgetter('started_at'))

    def get_tariff_snapshot(self):
        try:
            tariff_snapshot = self.tariff_snapshot
        except ObjectDoesNotExist:
            tariff_snapshot = None
        return tariff_snapshot
