import enum
import uuid

from django.db import models, transaction
from django.db.models import Prefetch
from django.utils import timezone

from cars.carsharing.models.acceptance import CarsharingAcceptance
from cars.carsharing.models.parking import CarsharingParking
from cars.carsharing.models.reservation import CarsharingReservation
from cars.carsharing.models.reservation_paid import CarsharingReservationPaid
from cars.carsharing.models.ride import CarsharingRide
from .order import Order
from .order_item_tariff import OrderItemTariff


class OrderItemManager(models.Manager):

    def with_related(self):
        from .order_item_payment import OrderItemPayment
        return (
            self.get_queryset()
            .select_related(
                'carsharing_acceptance__car__location',
                'carsharing_acceptance__car__model',
                'carsharing_acceptance__car__telematics_state',
                'carsharing_parking__car__location',
                'carsharing_parking__car__model',
                'carsharing_parking__car__telematics_state',
                'carsharing_reservation__car__location',
                'carsharing_reservation__car__model',
                'carsharing_reservation__car__telematics_state',
                'carsharing_reservation_paid__car__location',
                'carsharing_reservation_paid__car__model',
                'carsharing_reservation_paid__car__telematics_state',
                'carsharing_ride__car__location',
                'carsharing_ride__car__model',
                'carsharing_ride__car__telematics_state',
                'tariff__fix_params',
                'tariff__per_minute_params',
            )
            .prefetch_related(
                'carsharing_acceptance__photos',
                Prefetch('payments', queryset=OrderItemPayment.objects.with_related())
            )
        )

    def with_payments(self):
        from .order_item_payment import OrderItemPayment
        return (
            self.get_queryset()
            .select_related(
                'tariff__fix_params',
                'tariff__per_minute_params',
            )
            .prefetch_related(
                Prefetch(
                    'payments',
                    queryset=(
                        OrderItemPayment.objects
                        .select_related(
                            'bonus_payment',
                            'card_payment',
                        )
                    ),
                )
            )
        )


class OrderItem(models.Model):

    class Type(enum.Enum):
        CARSHARING_ACCEPTANCE = 'carsharing_acceptance'
        CARSHARING_PARKING = 'carsharing_parking'
        CARSHARING_RESERVATION = 'carsharing_reservation'
        CARSHARING_RESERVATION_PAID = 'carsharing_reservation_paid'
        CARSHARING_RIDE = 'carsharing_ride'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    type = models.CharField(max_length=32, choices=[(x.value, x.name) for x in Type])
    order = models.ForeignKey(
        Order,
        on_delete=models.CASCADE,
        related_name='items',
        db_index=False,
    )
    tariff = models.ForeignKey(
        OrderItemTariff,
        null=True,
        on_delete=models.SET_NULL,
        related_name='order_item',
        db_index=False,
    )
    started_at = models.DateTimeField()
    finished_at = models.DateTimeField(null=True)

    carsharing_acceptance = models.OneToOneField(
        CarsharingAcceptance,
        null=True,
        on_delete=models.CASCADE,
        related_name='order_item',
    )
    carsharing_parking = models.OneToOneField(
        CarsharingParking,
        null=True,
        on_delete=models.CASCADE,
        related_name='order_item',
    )
    carsharing_reservation = models.OneToOneField(
        CarsharingReservation,
        null=True,
        on_delete=models.CASCADE,
        related_name='order_item',
    )
    carsharing_reservation_paid = models.OneToOneField(
        CarsharingReservationPaid,
        null=True,
        on_delete=models.CASCADE,
        related_name='order_item',
    )
    carsharing_ride = models.OneToOneField(
        CarsharingRide,
        null=True,
        on_delete=models.CASCADE,
        related_name='order_item',
    )

    objects = OrderItemManager()

    class Meta:
        # pylint: disable=line-too-long
        db_table = 'order_item'
        db_constraints = {
            'exclusive_arc_chk': (
                '''
                CHECK (
                  (
                    (type = 'carsharing_acceptance' AND carsharing_acceptance_id IS NOT NULL)
                    OR
                    (type = 'carsharing_parking' AND carsharing_parking_id IS NOT NULL)
                    OR
                    (type = 'carsharing_reservation' AND carsharing_reservation_id IS NOT NULL)
                    OR
                    (type = 'carsharing_reservation_paid' AND carsharing_reservation_paid_id IS NOT NULL)
                    OR
                    (type = 'carsharing_ride' AND carsharing_ride_id IS NOT NULL)
                  )
                  AND
                  (
                    CASE WHEN carsharing_acceptance_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN carsharing_parking_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN carsharing_reservation_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN carsharing_reservation_paid_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN carsharing_ride_id IS NOT NULL THEN 1 ELSE 0 END
                  ) = 1
                )
                '''
            ),
        }
        indexes = [
            models.Index(
                fields=['order'],
                name='order_item_order_idx',
            ),
            models.Index(
                fields=['tariff'],
                name='order_item_tariff_idx',
            ),
            models.Index(
                fields=['started_at'],
                name='order_item_started_at_idx',
            ),
            models.Index(
                fields=['finished_at'],
                name='order_item_finished_at_idx',
            ),
        ]

    @property
    def duration(self):
        finish = self.finished_at if self.finished_at is not None else timezone.now()
        return finish - self.started_at

    def __repr__(self):
        return '<OrderItem: type={}, started_at={}, finished_at={}>'.format(
            self.type,
            self.started_at.isoformat(),
            self.finished_at.isoformat() if self.finished_at else None,
        )

    def get_type(self):
        return self.Type(self.type)

    def get_impl(self):
        type_ = self.get_type()
        if type_ is self.Type.CARSHARING_ACCEPTANCE:
            impl = self.carsharing_acceptance
        elif type_ is self.Type.CARSHARING_PARKING:
            impl = self.carsharing_parking
        elif type_ is self.Type.CARSHARING_RESERVATION:
            impl = self.carsharing_reservation
        elif type_ is self.Type.CARSHARING_RESERVATION_PAID:
            impl = self.carsharing_reservation_paid
        elif type_ is self.Type.CARSHARING_RIDE:
            impl = self.carsharing_ride
        else:
            raise RuntimeError('unreachable: {}'.format(type_))
        return impl

    def save(self, *args, **kwargs):
        with transaction.atomic():
            if self.tariff_id is None and self.tariff is not None:
                self.tariff.save(*args, **kwargs)
                self.tariff = self.tariff  # Make sure self.tariff_id is set.
            super().save(*args, **kwargs)
