import enum

from django.db import models

from cars.billing.models.bonus_payment import BonusPayment
from cars.billing.models.card_payment import CardPayment
from .order_item import OrderItem


class OrderItemPaymentManager(models.Manager):

    def with_related(self):
        return (
            self.get_queryset()
            .select_related(
                'bonus_payment',
                'card_payment',
            )
        )


class OrderItemPayment(models.Model):

    class PaymentMethod(enum.Enum):
        BONUS = 'bonus'
        CARD = 'card'

    order_item = models.ForeignKey(
        OrderItem,
        on_delete=models.CASCADE,
        db_index=False,
        related_name='payments',
    )
    payment_method = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in PaymentMethod],
    )
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    created_at = models.DateTimeField(null=True)

    bonus_payment = models.ForeignKey(
        BonusPayment,
        null=True,
        db_index=False,
        on_delete=models.CASCADE,
        related_name='order_item_payments',
    )

    card_payment = models.ForeignKey(
        CardPayment,
        null=True,
        db_index=False,
        on_delete=models.CASCADE,
        related_name='order_item_payments',
    )

    objects = OrderItemPaymentManager()

    class Meta:
        db_table = 'order_item_payment'
        db_constraints = {
            'exclusive_arc_chk': (
                '''
                CHECK (
                  (
                    (payment_method = 'bonus' AND bonus_payment_id IS NOT NULL)
                    OR
                    (payment_method = 'card' AND card_payment_id IS NOT NULL)
                  )
                  AND
                  (
                    CASE WHEN bonus_payment_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN card_payment_id IS NOT NULL THEN 1 ELSE 0 END
                  ) = 1
                )
                '''
            ),
        }
        indexes = [
            models.Index(
                fields=['order_item'],
                name='order_item_payment_order_item_idx',
            ),
            models.Index(
                fields=['bonus_payment'],
                name='order_item_payment_bonus_payment_idx',
            ),
            models.Index(
                fields=['card_payment'],
                name='order_item_payment_card_payment_idx',
            ),
        ]

    def __repr__(self):
        return '<OrderItemPayment: payment_method={}, amount={}>'.format(
            self.payment_method,
            self.amount,
        )

    def get_payment_method(self):
        return self.PaymentMethod(self.payment_method)

    def get_impl(self):
        impl = None
        payment_method = self.get_payment_method()
        if payment_method is self.PaymentMethod.BONUS:
            impl = self.bonus_payment
        elif payment_method is self.PaymentMethod.CARD:
            impl = self.card_payment
        else:
            raise RuntimeError('unreachable: {}'.format(payment_method))
        return impl
