import logging
import threading
import time

import shapely.geometry

import cars.settings
from cars.core.alerts import get_alerts_backend, get_alerts_listener
from cars.core.solomon import make_solomon_client
from cars.core.saas_index import SaasIndexListener
from cars.core.push_client import CarsharingPushClientLogger
from cars.core.pusher import Pusher


LOGGER = logging.getLogger(__name__)


class RadarDaemon(object):

    def __init__(self, saas_config, remind_interval):
        self._remind_interval = remind_interval

        self._alerts_backend = get_alerts_backend()

        self._alerts_listener = get_alerts_listener()
        self._alerts_listener.add_alert_updated_callback(self._on_alert_updated)

        self._car_index = SaasIndexListener(saas_config)
        self._car_index.add_car_updated_callback(self._on_car_updated)

        self._push_client = CarsharingPushClientLogger(
            filename=cars.settings.PUSH_CLIENT['radar']['filename'],
        )

        self._pusher = Pusher(push_client=self._push_client)

        self._solomon = make_solomon_client()

    def run(self):
        self._car_index.setup()
        self._alerts_listener.setup()

        index_listener_thread = threading.Thread(target=self._car_index.listen)
        index_listener_thread.daemon = True
        index_listener_thread.start()

        alerts_listener_thread = threading.Thread(target=self._alerts_listener.listen)
        alerts_listener_thread.daemon = True
        alerts_listener_thread.start()

        while True:
            if not index_listener_thread.is_alive():
                raise RuntimeError('Index listener crashed')
            if not alerts_listener_thread.is_alive():
                raise RuntimeError('Alerts listener crashed')

            now = time.time()
            for alert in self._alerts_listener.alerts:
                # Alert.reminded_at is None for new alerts.
                reminded_at = alert.reminded_at or alert.created_at
                if (now - reminded_at) > self._remind_interval:
                    try:
                        self._remind(alert)
                    except Exception:
                        LOGGER.exception('Failed to remind a user of an active alert')
                        continue

                    alert.reminded_at = now

                    try:
                        self._alerts_backend.upsert_alert(alert)
                    except Exception:
                        LOGGER.exception('Failed to upsert an alert')

            time.sleep(0.1)

    def _log(self, subtype, data):
        type_ = 'radar.{}'.format(subtype)
        self._push_client.log(type_=type_, data=data)

    def _remind(self, alert):
        self._log('push.reminder', {'alert': alert.to_dict()})
        try:
            self._pusher.send_reminder(alert.platform, alert.device_id, alert)
        except Exception:
            LOGGER.exception('Failed to send a reminder for %s', alert)
            return
        self._solomon.set_value('push_reminder', 1)

    def _on_alert_updated(self, old_alert, new_alert):
        if old_alert is None:
            self._log('alert.subscribe', {'alert': new_alert.to_dict()})
            self._solomon.set_value('alert_subscribe', 1)
        elif new_alert is None:
            self._log('alert.unsubscribe', {'alert': old_alert.to_dict()})
            self._solomon.set_value('alert_unsubscribe', 1)
        else:
            if old_alert == new_alert:
                # Watcher may be triggered by timestamp updates.
                # Ignore such modifications.
                return

            self._log('alert.update', {
                'old_alert': old_alert.to_dict(),
                'new_alert': new_alert.to_dict(),
            })
            self._solomon.set_value('alert_update', 1)

            for car in self._car_index.cars:

                is_old_filters_ok = old_alert.filter.check(car)
                is_new_filters_ok = new_alert.filter.check(car)

                # Check if the car satisfies new filters.
                if not is_new_filters_ok:
                    continue

                point = shapely.geometry.Point(car.lon, car.lat)

                # Skip the car if it is not within the new alert area.
                is_new_area_ok = new_alert.area.contains(point)
                if not is_new_area_ok:
                    continue

                # Skip the car if it is not new,
                # i.e. it has already been within the old alert area and passed the filters.
                if is_old_filters_ok:
                    # Slight optimization: don't check the area if the car is filtered out.
                    if old_alert.area.contains(point):
                        continue

                # All condictions are satisfied. Send the push.
                self._log('push.car', {'alert': new_alert.to_dict(), 'car': car.to_dict()})
                try:
                    self._pusher.send_free_car_push(
                        new_alert.platform, new_alert.device_id, new_alert.location, car,
                    )
                except Exception:
                    LOGGER.exception('Free car push failed')
                    continue
                self._solomon.set_value('push_car', 1)

    def _on_car_updated(self, old_car, new_car):
        if not new_car.is_free:
            # A car became rented.
            self._log('car.rent', {'car': new_car.to_dict()})
        else:
            # A car became free or changed its attributes (location, fuel, etc.).
            if old_car is None or not old_car.is_free:
                point = shapely.geometry.Point(new_car.lon, new_car.lat)
                for alert in self._alerts_listener.alerts:
                    if not alert.filter.check(new_car):
                        continue
                    if alert.area.contains(point):
                        self._log('push', {'alert': alert.to_dict(), 'car': new_car.to_dict()})
                        try:
                            self._pusher.send_free_car_push(
                                alert.platform, alert.device_id, alert.location, new_car,
                            )
                        except Exception:
                            LOGGER.exception('Free car push failed')
                            continue
                        self._solomon.set_value('push_car', 1)
                self._log('car.free', {'car': new_car.to_dict()})
            else:
                self._log('car.update', {
                    'old_car': old_car.to_dict(),
                    'new_car': new_car.to_dict(),
                })
