from rest_framework.exceptions import PermissionDenied

import cars.settings
from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from ..core.verifier import RefuelCarAccessVerifier


class RefuelAPIView(CarsharingAPIView):

    permission_classes = [IsAuthenticated]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._car_access_verifier = RefuelCarAccessVerifier.from_settings(
            allow_if_fueled=False,
            allow_if_available=True,
        )
        self._acl = cars.settings.REFUEL['acl']

    def get(self, request, *args, **kwargs):
        """
        Disallow viewing non-fueling car with >= specified percent of fuel.
        """
        self._check_access_authorization(request, *args, **kwargs)
        return super().get(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        """
        Disallow mutating operations on non-fueling cars with >= specified percent of fuel.
        """
        self._check_access_authorization(request, *args, **kwargs)
        return super().post(request, *args, **kwargs)

    def _check_access_authorization(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        if self._acl and request.user.uid not in self._acl:
            raise PermissionDenied

        car_id = kwargs.get('car_id')
        if self._car_access_verifier and not self._car_access_verifier.is_allowed(car_id):
            raise PermissionDenied
