import logging
import time
from abc import abstractmethod

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.models import Car
from cars.core.util import import_class
from ..core.verifier import RefuelCarAccessVerifier
from ..serializers.car import CarSerializer
from .base import RefuelAPIView


LOGGER = logging.getLogger(__name__)

TELEMATICS_COMMANDS_DELAY = cars.settings.REFUEL['telematics_commands_delay']
DELAY_BEFORE_WARMING = cars.settings.REFUEL['warming']['delay']


class CarCommandView(RefuelAPIView):

    lookup_url_kwarg = 'car_id'

    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def get_queryset(self):
        return Car.objects.all()

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = self.get_object()
        self.do_command(car)
        return Response()

    @abstractmethod
    def do_command(self, car):
        raise NotImplementedError


class OpenCarCommandView(CarCommandView):

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._car_access_verifier = RefuelCarAccessVerifier.from_settings(
            allow_if_fueled=False,  # don't allow to open car if it's fueled already
            allow_if_available=False,  # don't allow operations with car in AVAILABLE status
        )

    def do_command(self, car):
        self.telematics_proxy.open(car.imei)
        time.sleep(TELEMATICS_COMMANDS_DELAY)
        self.telematics_proxy.unlock_hood(car.imei)


class CarBlinkCommandView(CarCommandView):

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._car_access_verifier = RefuelCarAccessVerifier.from_settings(
            allow_if_fueled=False,  # don't allow to blink car if it's fueled already
            allow_if_available=False,  # don't allow operations with car in AVAILABLE status
        )

    def do_command(self, car):
        self.telematics_proxy.blink(car.imei)


class CloseCarCommandView(CarCommandView):

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._car_access_verifier = RefuelCarAccessVerifier.from_settings(
            allow_if_fueled=True,  # allow to close car if it is fueled
            allow_if_available=False,  # don't allow operations with car in AVAILABLE status
        )

    def do_command(self, car):
        self.telematics_proxy.lock_hood(car.imei)

        time.sleep(TELEMATICS_COMMANDS_DELAY)
        self.telematics_proxy.close(car.imei)

        time.sleep(DELAY_BEFORE_WARMING)
        try:
            response = self.telematics_proxy.warmup(car.imei)
            response.raise_for_status()
        except Exception:
            LOGGER.exception('failed to warm up car with imei=%s', str(car.imei))
        else:
            LOGGER.info('successfully launched warmup for car with imei=%s', str(car.imei))


class CarStatusView(RefuelAPIView):

    lookup_url_kwarg = 'car_id'

    serializer_class = CarSerializer

    def get_queryset(self):
        return Car.objects.all()

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = self.get_object()

        car_updater = CarUpdater(car)
        try:
            self.do_update_status(car_updater, car)
        except car_updater.BadStatusError:
            return Response(status=HTTP_400_BAD_REQUEST)

        return Response(self.serializer_class(self.get_object()).data)

    @abstractmethod
    def do_update_status(self, car_updater, car):
        raise NotImplementedError


class MoveToAvailableCarStatusView(CarStatusView):

    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def do_update_status(self, car_updater, car):
        if car.status != Car.Status.SERVICE.value:
            car_updater.update_status(Car.Status.AVAILABLE)


class MoveToFuelingCarStatusView(CarStatusView):

    def do_update_status(self, car_updater, car):
        if car.status != Car.Status.SERVICE.value:
            car_updater.update_status(Car.Status.FUELING)
