from django.db.models import Q
from rest_framework.mixins import ListModelMixin, RetrieveModelMixin
from rest_framework.response import Response

import cars.settings
from cars.carsharing.models import Car
from ..serializers.car import CarSerializer
from .base import RefuelAPIView


class CarListView(ListModelMixin, RefuelAPIView):

    serializer_class = CarSerializer

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._car_access_verifier = None

    def do_get(self, _):
        refuel_threshold = cars.settings.REFUEL['refuel_threshold']
        car_objects = (
            Car.objects
            .filter(
                Q(status=Car.Status.FUELING.value) | Q(
                    status__in=[
                        Car.Status.AVAILABLE.value,
                        Car.Status.SERVICE.value,
                    ],
                    telematics_state__fuel_level__lte=refuel_threshold,
                )
            )
            .select_related('location', 'model', 'telematics_state')
        )
        data = {
            'cars': CarSerializer(car_objects, many=True).data,
        }
        return Response(data)


class CarDetailsView(RetrieveModelMixin, RefuelAPIView):

    lookup_url_kwarg = 'car_id'
    serializer_class = CarSerializer

    def get_queryset(self):
        return (
            Car.objects
            .select_related('location', 'model', 'telematics_state')
        )

    def do_get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)
