import json
import logging
import random

from django.utils import timezone

import cars.settings
from cars.core.mds.wrapper import MDSDocumentsWrapper
from cars.core.solomon import SOLOMON
from cars.core.util import import_class
from cars.users.core import UserProfileUpdater
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.core.recognized_data_submitter import RecognizedDataSubmitter
from ...models import RegistrationChatActionResult, RegistrationVerificationCall
from ...serializers.chat_actions import (
    CreditCardChatActionDataSerializer,
    DriverLicenseBackChatActionDataSerializer,
    DriverLicenseFrontChatActionDataSerializer,
    DriverLicenseChatActionDataSerializer,
    PassportBiographicalChatActionDataSerializer,
    PassportChatActionDataSerializer,
    PassportRegistrationChatActionDataSerializer,
    PassportSelfieChatActionDataSerializer,
    PhotoChatActionDataSerializer,
)


LOGGER = logging.getLogger(__name__)


class BaseChatAction(object):

    class ValidationError(Exception):
        pass

    Type = RegistrationChatActionResult.Type

    data_serializer_class = None

    mds_client = MDSDocumentsWrapper.from_settings()

    solomon_client = SOLOMON

    def __init__(self, *, id_):
        self._id = id_

    @property
    def id(self):
        return self._id

    def get_type(self):
        raise NotImplementedError

    def get_params(self):
        raise NotImplementedError

    def submit(self, user, action_result, data):
        validated_data = self._validate(data=data)
        return self._submit_validated(
            user=user,
            validated_data=validated_data,
            action_result=action_result,
        )

    def _submit_validated(self, user, validated_data, action_result):
        """
        Returns ChatActionSubmissionResult instance.

        Should be idempotent.
        """
        raise NotImplementedError

    def _validate(self, data):
        serializer = self.data_serializer_class(data=data)  # pylint: disable=not-callable
        if not serializer.is_valid():
            raise self.ValidationError(serializer.errors)
        return serializer.validated_data


class ChatActionSubmissionResult(object):

    def __init__(self, data, status=RegistrationChatActionResult.Status.COMPLETE):
        self.status = status
        self.data = data


class OkChatAction(BaseChatAction):

    def __init__(self, *args, text, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = text

    def get_type(self):
        return self.Type.OK

    def get_params(self):
        return {
            'text': self.text,
        }

    def _submit_validated(self, user, validated_data, action_result):
        return ChatActionSubmissionResult(data={})

    def _validate(self, data):
        if data:
            raise self.ValidationError('no_data_expected')
        return data


class CallVerifyChatAction(OkChatAction):

    _octopus_client = None

    @classmethod
    def _get_octopus_client(cls):
        if cls._octopus_client is None:
            octopus_client = import_class(cars.settings.OCTOPUS['client_class']).from_settings()
            with open(cars.settings.REGISTRATION['octopus']['script_path']) as f:
                script = json.load(f)

            prompt = random.choice(cars.settings.REGISTRATION['octopus']['prompts'])
            script['steps']['ask_phrase']['parameters']['prompt'] = prompt['question']
            script['steps']['say_thanks']['parameters']['prompt'] = prompt['thanks']

            prepared_client = octopus_client.prepare(script=script)
            cls._octopus_client = prepared_client

        return cls._octopus_client

    def get_type(self):
        return self.Type.CALL_VERIFY

    def _get_octopus_session_fail_solomon_sensor(self):
        return 'registration.octopus_session_fail'

    def _submit_validated(self, user, validated_data, action_result):
        assert user.phone is not None, 'calling to a user without a phone: {}'.format(user.id)
        client = self._get_octopus_client()
        try:
            session_id = client.call(number=user.phone.as_e164)
            call_status = RegistrationVerificationCall.Status.INPROGRESS
            action_status = RegistrationChatActionResult.Status.INPROGRESS
        except Exception:
            LOGGER.exception('Failed to create octopus session.')
            #  skip registration step
            session_id = None
            call_status = RegistrationVerificationCall.Status.OK
            action_status = RegistrationChatActionResult.Status.COMPLETE
            try:
                self.solomon_client.set_value(
                    self._get_octopus_session_fail_solomon_sensor(),
                    1
                )
            except Exception:
                LOGGER.exception('Failed to send solomon data')
        RegistrationVerificationCall.objects.create(
            user=user,
            chat_action_result=action_result,
            session_id=session_id,
            submitted_at=timezone.now(),
            status=call_status.value,
        )
        return ChatActionSubmissionResult(
            status=action_status,
            data={},
        )


class CreditCardChatAction(BaseChatAction):

    data_serializer_class = CreditCardChatActionDataSerializer

    def __init__(self, *args, text, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = text

    def get_type(self):
        return self.Type.CREDIT_CARD

    def get_params(self):
        return {
            'text': self.text,
        }

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user)
        updater.update_credit_card(
            pan_prefix=validated_data['pan']['prefix'],
            pan_suffix=validated_data['pan']['suffix'],
        )
        return ChatActionSubmissionResult(data=validated_data)


class PhotoChatAction(BaseChatAction):

    data_serializer_class = PhotoChatActionDataSerializer

    def __init__(self, *args, text, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = text

    def get_params(self):
        return {
            'text': self.text,
        }

    def get_type(self):
        raise NotImplementedError

    def _submit_validated(self, user, validated_data, action_result):
        pass


class DriverLicenseChatAction(BaseChatAction):

    data_serializer_class = DriverLicenseChatActionDataSerializer

    datasync_client = DataSyncDocumentsClient.from_settings()
    recognized_data_submitter = RecognizedDataSubmitter(datasync_client)

    def __init__(self, *args, text, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = text

    def get_params(self):
        return {
            'text': self.text,
        }

    def get_type(self):
        return self.Type.DRIVER_LICENSE

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(
            user=user,
            mds_client=self.mds_client,
            recognized_data_submitter=self.recognized_data_submitter,
        )

        front = validated_data['front']
        back = validated_data['back']
        front_photo, back_photo = updater.update_unverified_driver_license(
            front_content=front['content'],
            front_recognized=front['scanners'],
            back_content=back['content'],
            back_recognized=back['scanners'],
        )

        data = {
            'front': {
                'document_id': str(front_photo.document_id),
                'photo_id': str(front_photo.id),
                'type': front_photo.type,
            },
            'back': {
                'document_id': str(back_photo.document_id),
                'photo_id': str(back_photo.id),
                'type': back_photo.type,
            },
        }

        return ChatActionSubmissionResult(data=data)


class DriverLicenseBackChatAction(PhotoChatAction):

    data_serializer_class = DriverLicenseBackChatActionDataSerializer

    def get_type(self):
        return self.Type.DRIVER_LICENSE_BACK

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user, mds_client=self.mds_client)
        back_photo = updater.update_unverified_driver_license_back(
            content=validated_data['content'],
            recognized=validated_data['scanners'],
        )
        data = {
            'document_id': str(back_photo.document_id),
            'photo_id': str(back_photo.id),
            'type': back_photo.type,
        }
        return ChatActionSubmissionResult(data=data)


class DriverLicenseFrontChatAction(PhotoChatAction):

    data_serializer_class = DriverLicenseFrontChatActionDataSerializer

    def get_type(self):
        return self.Type.DRIVER_LICENSE_FRONT

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user, mds_client=self.mds_client)
        back_photo = updater.update_unverified_driver_license_front(
            content=validated_data['content'],
            recognized=validated_data['scanners'],
        )
        data = {
            'document_id': str(back_photo.document_id),
            'photo_id': str(back_photo.id),
            'type': back_photo.type,
        }
        return ChatActionSubmissionResult(data=data)


class PassportChatAction(BaseChatAction):

    data_serializer_class = PassportChatActionDataSerializer

    datasync_client = DataSyncDocumentsClient.from_settings()
    recognized_data_submitter = RecognizedDataSubmitter(datasync_client)

    def __init__(self, *args, text, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = text

    def get_params(self):
        return {
            'text': self.text,
        }

    def get_type(self):
        return self.Type.PASSPORT

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(
            user=user,
            mds_client=self.mds_client,
            recognized_data_submitter=self.recognized_data_submitter,
        )

        biographical = validated_data['biographical']
        registration = validated_data['registration']
        biographical_photo, registration_photo = updater.update_unverified_passport(
            biographical_content=biographical['content'],
            biographical_recognized=biographical['scanners'],
            registration_content=registration['content'],
            registration_recognized=registration['scanners'],
        )

        data = {
            'biographical': {
                'document_id': str(biographical_photo.document_id),
                'photo_id': str(biographical_photo.id),
                'type': biographical_photo.type,
            },
            'registration': {
                'document_id': str(registration_photo.document_id),
                'photo_id': str(registration_photo.id),
                'type': registration_photo.type,
            },
        }

        return ChatActionSubmissionResult(data=data)


class PassportBiographicalChatAction(PhotoChatAction):

    data_serializer_class = PassportBiographicalChatActionDataSerializer

    def get_type(self):
        return self.Type.PASSPORT_BIOGRAPHICAL

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user, mds_client=self.mds_client)
        photo = updater.update_unverified_passport_biographical(
            content=validated_data['content'],
            recognized=validated_data['scanners'],
        )
        data = {
            'document_id': str(photo.document_id),
            'photo_id': str(photo.id),
            'type': photo.type,
        }
        return ChatActionSubmissionResult(data=data)


class PassportRegistrationChatAction(PhotoChatAction):

    data_serializer_class = PassportRegistrationChatActionDataSerializer

    def get_type(self):
        return self.Type.PASSPORT_REGISTRATION

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user, mds_client=self.mds_client)
        photo = updater.update_unverified_passport_registration(
            content=validated_data['content'],
            recognized=validated_data['scanners'],
        )
        data = {
            'document_id': str(photo.document_id),
            'photo_id': str(photo.id),
            'type': photo.type,
        }
        return ChatActionSubmissionResult(data=data)


class PassportSelfieChatAction(PhotoChatAction):

    data_serializer_class = PassportSelfieChatActionDataSerializer

    def get_type(self):
        return self.Type.PASSPORT_SELFIE

    def _submit_validated(self, user, validated_data, action_result):
        updater = UserProfileUpdater(user=user, mds_client=self.mds_client)
        face_data = validated_data['face']
        _, selfie_photo = updater.update_unverified_passport_selfie(
            face_content=face_data['content'] if face_data else None,
            selfie_content=validated_data['selfie']['content'],
        )
        data = {
            'document_id': str(selfie_photo.document_id),
            'photo_id': str(selfie_photo.id),
            'type': selfie_photo.type,
        }
        return ChatActionSubmissionResult(data=data)


class EnterAppChatAction(OkChatAction):

    def get_type(self):
        return self.Type.ENTER_APP

    def _submit_validated(self, user, validated_data, action_result):
        return ChatActionSubmissionResult(
            status=RegistrationChatActionResult.Status.NEW,
            data={},
        )
