import logging

from django.utils import timezone

import cars.settings
from ...models.chat_action_result import RegistrationChatActionResult
from ...models.chat_message import RegistrationChatMessage
from .actions import EnterAppChatAction
from .items import RetryActionChatItem


LOGGER = logging.getLogger(__name__)


class ChatBuilder(object):

    def __init__(self, script, context):
        self._script = script
        self._context = context

    def build_action_messages(self, user, action, action_result):
        action_type = RegistrationChatActionResult.Type(action_result.type)

        if action_type in (RegistrationChatActionResult.Type.CALL_VERIFY,
                           RegistrationChatActionResult.Type.OK):
            chat_message_types = RegistrationChatMessage.Type.TEXT
            chat_message_contents = {
                'text': action.text,
            }
        elif action_type == RegistrationChatActionResult.Type.CREDIT_CARD:
            chat_message_types = RegistrationChatMessage.Type.CARD
            chat_message_contents = {
                'pan': {
                    'prefix': action_result.data['pan']['prefix'],
                    'suffix': action_result.data['pan']['suffix'],
                },
            }
        elif action_type in (RegistrationChatActionResult.Type.DRIVER_LICENSE_BACK,
                             RegistrationChatActionResult.Type.DRIVER_LICENSE_FRONT,
                             RegistrationChatActionResult.Type.PASSPORT_BIOGRAPHICAL,
                             RegistrationChatActionResult.Type.PASSPORT_REGISTRATION,
                             RegistrationChatActionResult.Type.PASSPORT_SELFIE):
            chat_message_types = RegistrationChatMessage.Type.USER_DOCUMENT
            chat_message_contents = action_result.data
        elif action_type is RegistrationChatActionResult.Type.DRIVER_LICENSE:
            chat_message_types = RegistrationChatMessage.Type.USER_DOCUMENTS
            chat_message_contents = {
                'docs': [
                    action_result.data['front'],
                    action_result.data['back'],
                ],
            }
        elif action_type is RegistrationChatActionResult.Type.PASSPORT:
            chat_message_types = RegistrationChatMessage.Type.USER_DOCUMENTS
            chat_message_contents = {
                'docs': [
                    action_result.data['biographical'],
                    action_result.data['registration'],
                ],
            }
        else:
            LOGGER.error('unknown chat action result type: %s', action_type)
            raise RuntimeError('unreachable: {}'.format(action_type))

        if not isinstance(chat_message_types, list):
            chat_message_types = [chat_message_types]
            chat_message_contents = [chat_message_contents]

        chat_messages = []
        for type_, content in zip(chat_message_types, chat_message_contents):
            chat_message = RegistrationChatMessage(
                user=user,
                date=timezone.now(),
                chat_action_id=action.id,
                source=RegistrationChatMessage.Source.USER.value,
                type=type_.value,
                content=content,
            )
            chat_messages.append(chat_message)

        return chat_messages

    def run_action(self, user, chat_action_id, context=None):
        group = self._script.get_action_group_by_id(chat_action_id)
        action = group.action
        new_messages = self._build_pre_action_messages(
            user=user,
            action=action,
            pre_action=group.pre_action,
            context=context,
        )
        return action, new_messages

    def run_until_next_action(self, user, current_action_id, context=None):
        action = None
        new_messages = []

        # Finish any existing action.
        if current_action_id is not None:
            group = self._script.get_action_group_by_id(current_action_id)
            action, post_action_messages = self._build_post_action_messages(
                user=user,
                action=group.action,
                post_action=group.post_action,
                context=context,
            )
            new_messages += post_action_messages

        # Find the next available action according to the flow.
        if action is None:
            found_current_action = current_action_id is None
            for flow_item in self._script.flow:
                group = self._script.get_action_group_by_id(flow_item.action_group_id)

                # Skip call and credit card steps for fake users.
                if user.uid in cars.settings.REGISTRATION['fake_uids']:
                    if group.action.id in {'call', 'credit_card'}:
                        continue

                if group.action.id == current_action_id:
                    found_current_action = True
                    continue

                if found_current_action:
                    action = group.action
                    new_messages += self._build_pre_action_messages(
                        user=user,
                        action=action,
                        pre_action=group.pre_action,
                        context=context,
                    )
                    break

        return action, new_messages

    def _build_pre_action_messages(self, user, action, pre_action, context=None):
        new_action, messages = self._build_chat_items_messages(
            user=user,
            action=action,
            chat_items=pre_action.chat_items,
            context=context,
        )
        assert new_action is None
        return messages

    def _build_post_action_messages(self, user, action, post_action, context=None):
        new_action, messages = self._build_chat_items_messages(
            user=user,
            action=action,
            chat_items=post_action.chat_items,
            context=context,
        )
        return new_action, messages

    def _build_chat_items_messages(self, user, action, chat_items, context=None):
        messages = []

        for chat_item in chat_items:
            if chat_item.when is not None:
                ok = self._eval_when_clause(chat_item.when, context=context)
                if not ok:
                    continue

            if isinstance(chat_item, RetryActionChatItem):
                return action, messages

            chat_item_messages = self._build_chat_item_messages(
                user=user,
                chat_action_id=action.id,
                chat_item=chat_item,
                context=context,
            )
            messages += chat_item_messages

        return None, messages

    def _eval_when_clause(self, when, context):
        if isinstance(when, dict):
            result = self._eval_when_dict(when, context)
        else:
            assert isinstance(when, str)
            result = self._eval_when_str(when, context)
        return result

    def _eval_when_dict(self, when, context):
        assert len(when) == 1

        cond_name, cond_val = list(when.items())[0]

        if cond_name == 'eq':
            assert isinstance(cond_val, list)
            assert len(cond_val) == 2
            symbol1, symbol2 = cond_val

            try:
                context_value1 = self._context.get(symbol1)
            except self._context.ScriptContextKeyError:
                context_value1 = None
            try:
                context_value2 = self._context.get(symbol2)
            except self._context.ScriptContextKeyError:
                context_value2 = None

            if context_value1 is None and context_value2 is None:
                return False

            value1 = context_value1 if context_value1 is not None else symbol1
            value2 = context_value2 if context_value2 is not None else symbol2

            return value1 == value2
        elif cond_name == 'not':
            return not self._eval_when_clause(cond_val, context)
        else:
            raise RuntimeError('unreachable: {}'.format(cond_name))

    def _eval_when_str(self, when, context):
        context = self._context.update_from_dict(context) if context else self._context
        value = context.get(when)
        return bool(value)

    def build_message_group_messages(self, user, message_group, context=None):
        messages = []
        for chat_item in message_group.chat_items:
            chat_item_messages = self._build_chat_item_messages(
                user=user,
                chat_action_id=message_group.id,
                chat_item=chat_item,
                context=context,
            )
            messages += chat_item_messages
        return messages

    def _build_chat_item_messages(self, user, chat_action_id, chat_item, context):
        chat_item_type = chat_item.get_type()
        messages = []

        if chat_item.when is not None:
            ok = self._eval_when_clause(chat_item.when, context=context)
            if not ok:
                return []

        if context:
            context = self._context.update_from_dict(context)
        else:
            context = self._context

        if chat_item_type is chat_item.Type.TEXT:
            text = context.expand(chat_item.message)
            message = RegistrationChatMessage(
                user=user,
                date=timezone.now(),
                chat_action_id=chat_action_id,
                source=RegistrationChatMessage.Source.SYSTEM.value,
                type=RegistrationChatMessage.Type.TEXT.value,
                on_tap=chat_item.on_tap,
                content={
                    'text': text,
                },
            )
            messages.append(message)

        elif chat_item_type is chat_item.Type.TEXT_BLOCK:
            for chat_item_message in chat_item.messages:
                text = context.expand(chat_item_message)
                message = RegistrationChatMessage(
                    user=user,
                    date=timezone.now(),
                    chat_action_id=chat_action_id,
                    source=RegistrationChatMessage.Source.SYSTEM.value,
                    type=RegistrationChatMessage.Type.TEXT.value,
                    content={
                        'text': text,
                    },
                )
                messages.append(message)

        elif chat_item_type is chat_item.Type.IMAGE:
            message = RegistrationChatMessage(
                user=user,
                date=timezone.now(),
                chat_action_id=chat_action_id,
                source=RegistrationChatMessage.Source.SYSTEM.value,
                type=RegistrationChatMessage.Type.IMAGE.value,
                on_tap=chat_item.on_tap,
                content={
                    'url': chat_item.url,
                },
            )
            messages.append(message)

        else:
            raise RuntimeError('unreachable')

        return messages

    def create_enter_app_action(self):
        return EnterAppChatAction(id_='enter_app', text='Поехали!')

    def create_fake_action(self, user):
        action = self.create_enter_app_action()
        messages = [
            RegistrationChatMessage(
                id=94312022,
                user=user,
                date=timezone.now(),
                chat_action_id=action.id,
                source=RegistrationChatMessage.Source.SYSTEM.value,
                type=RegistrationChatMessage.Type.TEXT.value,
                on_tap=None,
                content={
                    'text': 'Регистрация пройдена.',
                },
            ),
            RegistrationChatMessage(
                id=94312023,
                user=user,
                date=timezone.now(),
                chat_action_id=action.id,
                source=RegistrationChatMessage.Source.SYSTEM.value,
                type=RegistrationChatMessage.Type.TEXT.value,
                on_tap=None,
                content={
                    'text': 'Поздравляю!',
                },
            ),
        ]
        return action, messages
