import copy
import logging
import operator
import re

from ...models.chat_action_result import RegistrationChatActionResult
from ...models.verification_call import RegistrationVerificationCall


LOGGER = logging.getLogger(__name__)


class ScriptContext(object):

    class ScriptContextKeyError(Exception):
        pass

    def __init__(self, context):
        self._context = context

    def __getitem__(self, key):
        return self._context[key]

    @classmethod
    def from_action_results(cls, action_results):
        context = {}

        action_results = sorted(
            action_results, key=operator.attrgetter('suggested_at'),
        )
        if action_results:
            latest_action_result = action_results[-1]
            cls.update_context_with_latest_action_result(context, latest_action_result)

        return cls(context=context)

    @classmethod
    def update_context_with_latest_action_result(cls, context, action_result):
        if action_result.type == RegistrationChatActionResult.Type.CALL_VERIFY.value:
            try:
                call = RegistrationVerificationCall.objects.get(chat_action_result=action_result)
                call_status = RegistrationVerificationCall.Status(call.status).name
            except RegistrationVerificationCall.DoesNotExist:
                call_status = None
            context['call_status'] = call_status

    def get(self, key):
        value = self
        attr_path = key.split('.')
        for attr in attr_path:
            try:
                value = self._get_context_value(value, attr)
            except (AttributeError, KeyError):
                raise self.ScriptContextKeyError(attr)
        return value

    def update_from_dict(self, data):
        merged_context = copy.deepcopy(self._context)
        merged_context.update(data)
        return type(self)(merged_context)

    def expand(self, string):
        """Substitute values from the context into a string"""

        context_keys = re.findall('{(.+?)}', string)

        for key in context_keys:
            value = self.get(key)
            string = string.replace('{{{}}}'.format(key), str(value))

        return string

    def _get_context_value(self, obj, key):
        try:
            value = obj[key]
        except TypeError:
            value = getattr(obj, key)
        return value
