import datetime
import io
import logging
import operator
import os

import pytz
import yaml
from django.db import transaction
from django.utils import timezone

import cars.settings
from cars.core.constants import AppPlatform
from cars.users.models.app_install import AppInstall
from cars.users.models.user import User
from ...models.chat_action_queue import RegistrationChatActionQueue
from ...models.chat_action_result import RegistrationChatActionResult
from ...models.chat_message import RegistrationChatMessage
from .builder import ChatBuilder
from .context import ScriptContext
from .script import RegistrationScript
from .state import ChatState


LOGGER = logging.getLogger(__name__)


class ChatManager(object):

    def __init__(self, raw_script=None):
        if raw_script is None:
            raw_script = {
                'flow': [],
                'action_groups': [],
                'message_groups': [],
            }
        self._raw_script = raw_script

    @classmethod
    def from_settings(cls):
        script_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)),
            'resources',
            'script.yml',
        )
        obj = cls.from_yaml_file(path=script_path)
        return obj

    @classmethod
    def from_yaml(cls, content):
        raw_script = yaml.load(io.StringIO(content))
        return cls(raw_script=raw_script)

    @classmethod
    def from_yaml_file(cls, path):
        with open(path, encoding='utf-8') as f:
            content = f.read()
        return cls.from_yaml(content=content)

    def make_session(self, user, context=None):
        context_obj = self._load_context(user=user)
        if context:
            context_obj = context_obj.update_from_dict(context)

        script = RegistrationScript.from_dict(
            raw_script=self._raw_script,
            context=context_obj,
        )
        chat_builder = ChatBuilder(
            script=script,
            context=context_obj,
        )
        session = ChatManagerSession(
            user=user,
            script=script,
            chat_builder=chat_builder,
        )
        return session

    def get_chat_state(self, user, allow_initialize=True,
                       since=pytz.utc.localize(datetime.datetime.min)):
        session = self.make_session(user=user)
        return session.get_chat_state(since=since, allow_initialize=allow_initialize)

    def submit_chat_action(self, user, chat_action_id, data):
        session = self.make_session(user=user)
        try:
            with transaction.atomic(savepoint=False):
                # Lock the user to avoid concurrent updates.
                User.objects.select_for_update().get(id=user.id)
                action_result, action_messages = session.submit_chat_action(
                    chat_action_id=chat_action_id,
                    data=data,
                )
                if action_result.is_status_complete():
                    session = self.make_session(user=user)
                    new_action, new_action_messages = session.run_until_next_action(
                        current_action_id=chat_action_id,
                        context=action_result.context,
                    )
                else:
                    new_action = None
                    new_action_messages = []
        except session.ActionAlreadySubmitted as e:
            # This may happen due to balancer retries.
            # Return chat history from that moment.
            LOGGER.warning('tried to resubmit an action: %s', chat_action_id)
            state = self.get_chat_state(user=user, since=e.action_result.submitted_at)
        else:
            new_messages = action_messages + new_action_messages
            state = ChatState(action=new_action, new_messages=new_messages)

        return state

    def complete_chat_action(self, chat_action_result, need_atomicity=True):
        """Complete chat action in INPROGRESS state."""
        session = self.make_session(chat_action_result.user)
        if need_atomicity:
            with transaction.atomic(savepoint=False):
                chat_action_result = (
                    RegistrationChatActionResult.objects
                    .select_for_update()
                    .get(id=chat_action_result.id)
                )
                assert chat_action_result.status == RegistrationChatActionResult.Status.INPROGRESS.value
                chat_action_result.status = RegistrationChatActionResult.Status.COMPLETE.value
                chat_action_result.completed_at = timezone.now()
                chat_action_result.save()
                session.run_until_next_action(current_action_id=chat_action_result.chat_action_id, need_atomicity=False)
        else:
            chat_action_result = (
                RegistrationChatActionResult.objects
                    .select_for_update()
                    .get(id=chat_action_result.id)
            )
            assert chat_action_result.status == RegistrationChatActionResult.Status.INPROGRESS.value
            chat_action_result.status = RegistrationChatActionResult.Status.COMPLETE.value
            chat_action_result.completed_at = timezone.now()
            chat_action_result.save()
            session.run_until_next_action(current_action_id=chat_action_result.chat_action_id, need_atomicity=False)

    def _load_context(self, user):
        action_results = list(
            RegistrationChatActionResult.objects
            .filter(user=user)
            .order_by('suggested_at')
        )
        context = ScriptContext.from_action_results(action_results=action_results)
        return context


class ChatManagerSession(object):

    class ActionAlreadySubmitted(Exception):

        def __init__(self, *args, action_result, **kwargs):
            super().__init__(*args, **kwargs)
            self.action_result = action_result

    class ActionNotFound(Exception):
        pass

    def __init__(self, user, script, chat_builder):
        self._user = user
        self._script = script
        self._chat_builder = chat_builder

    def get_chat_state(self, allow_initialize=True, since=pytz.utc.localize(datetime.datetime.min)):
        action_results = list(
            RegistrationChatActionResult.objects
            .filter(user=self._user)
            .order_by('suggested_at')
        )
        messages = list(
            RegistrationChatMessage.objects
            .filter(
                user=self._user,
                date__gt=since,
                is_visible=True,
            )
            .order_by('date')
        )

        if not action_results:
            if allow_initialize:
                # It is the first chat request from this user.
                action, new_messages = self._initialize_chat()
                messages = sorted(messages + new_messages, key=operator.attrgetter('date'))
            else:
                action = None
        elif action_results[-1].is_status_new():
            action_result = action_results[-1]
            action = self._script.get_chat_action_by_id(
                action_group_id=action_result.chat_action_id,
            )
        else:
            action = None

        if action is None:
            action, messages_from_queue = self._build_action_from_queue()
            messages += messages_from_queue

        if (self._user.uid in cars.settings.REGISTRATION['fake_uids']
                and action is None
                and messages
                and messages[-1].source == RegistrationChatMessage.Source.USER.value
                and messages[-1].chat_action_id == 'passport_selfie'):
            action, new_messages = self._chat_builder.create_fake_action(self._user)
            messages += new_messages

        if action is None:
            action, new_messages = self._build_final_action()
            messages += new_messages

        state = ChatState(action=action, new_messages=messages)

        return state

    def clear_chat(self, force=False):
        if force:
            (
                RegistrationChatActionResult.objects
                    .filter(
                    user=self._user,
                    completed_at__isnull=True,
                )
                .all()
                .delete()
            )

            (
                RegistrationChatActionQueue.objects
                .filter(
                    user=self._user,
                    dequeued_at__isnull=True,
                )
                .all()
                .delete()
            )

        has_actions_in_chat = (
            RegistrationChatActionResult.objects
            .filter(
                user=self._user,
                completed_at__isnull=True,
            )
            .exists()
        )
        has_actions_in_queue = (
            RegistrationChatActionQueue.objects
            .filter(
                user=self._user,
                dequeued_at__isnull=True,
            )
            .exists()
        )

        assert not has_actions_in_chat
        assert not has_actions_in_queue

        RegistrationChatMessage.objects.filter(user=self._user).update(is_visible=False)

    def reset_chat(self):
        with transaction.atomic(savepoint=False):
            RegistrationChatActionQueue.objects.filter(user=self._user).delete()
            RegistrationChatActionResult.objects.filter(user=self._user).delete()
            RegistrationChatMessage.objects.filter(user=self._user).delete()

    def enqueue_chat_action(self, chat_action_id, context=None):
        RegistrationChatActionQueue.objects.create(
            user=self._user,
            chat_action_id=chat_action_id,
            context=context,
            enqueued_at=timezone.now(),
        )

    def submit_chat_action(self, chat_action_id, data):
        action_results = list(
            RegistrationChatActionResult.objects
            .filter(
                user=self._user,
                chat_action_id=chat_action_id,
            )
            .order_by('suggested_at')
        )
        if not action_results:
            raise self.ActionNotFound(chat_action_id)
        action_result = action_results[-1]

        if action_result.status == RegistrationChatActionResult.Status.INPROGRESS.value:
            return action_result, []

        if action_result.status != RegistrationChatActionResult.Status.NEW.value:
            raise self.ActionAlreadySubmitted('already_submitted', action_result=action_result)

        action = self._script.get_chat_action_by_id(action_group_id=chat_action_id)
        if action is None:
            raise self.ActionNotFound('action_not_found')

        submission_result = action.submit(
            user=self._user,
            action_result=action_result,
            data=data
        )

        if submission_result.status is RegistrationChatActionResult.Status.NEW:
            return action_result, []

        action_result.status = submission_result.status.value
        action_result.data = submission_result.data
        action_result.submitted_at = timezone.now()
        if submission_result.status is RegistrationChatActionResult.Status.COMPLETE:
            action_result.completed_at = action_result.submitted_at
        action_result.save()

        action_messages = self._chat_builder.build_action_messages(
            user=self._user,
            action=action,
            action_result=action_result,
        )
        RegistrationChatMessage.objects.bulk_create(action_messages)

        return action_result, action_messages

    def run_until_next_action(self, current_action_id, context=None, need_atomicity=True):
        if need_atomicity:
            with transaction.atomic(savepoint=False):
                new_action, new_messages = self._chat_builder.run_until_next_action(
                    user=self._user,
                    current_action_id=current_action_id,
                    context=context,
                )

                if new_action or new_messages:
                    self._create_action_with_messages(action=new_action, messages=new_messages, need_atomicity=False)

                if new_action is None:
                    new_action, new_messages_from_queue = self._build_action_from_queue(need_atomicity=False)
                    new_messages += new_messages_from_queue

                registration_state = self._user.get_registration_state()
                if registration_state.chat_started_at is None:
                    registration_state.chat_started_at = timezone.now()
                if new_action is not None:
                    registration_state.chat_action_id = new_action.id
                else:
                    registration_state.chat_action_id = None
                    if registration_state.chat_completed_at is None:
                        registration_state.chat_completed_at = timezone.now()
                registration_state.save()
        else:
            new_action, new_messages = self._chat_builder.run_until_next_action(
                user=self._user,
                current_action_id=current_action_id,
                context=context,
            )

            if new_action or new_messages:
                self._create_action_with_messages(action=new_action, messages=new_messages, need_atomicity=False)

            if new_action is None:
                new_action, new_messages_from_queue = self._build_action_from_queue(need_atomicity=False)
                new_messages += new_messages_from_queue

            registration_state = self._user.get_registration_state()
            if registration_state.chat_started_at is None:
                registration_state.chat_started_at = timezone.now()
            if new_action is not None:
                registration_state.chat_action_id = new_action.id
            else:
                registration_state.chat_action_id = None
                if registration_state.chat_completed_at is None:
                    registration_state.chat_completed_at = timezone.now()
            registration_state.save()

        return new_action, new_messages

    def _build_action_from_queue(self, need_atomicity=True):
        enqueued_action = (
            RegistrationChatActionQueue.objects
            .filter(
                user=self._user,
                dequeued_at__isnull=True,
            )
            .order_by('enqueued_at')
            .first()
        )

        if enqueued_action is None:
            return None, []

        new_action, new_messages = self._chat_builder.run_action(
            user=self._user,
            chat_action_id=enqueued_action.chat_action_id,
            context=enqueued_action.context,
        )

        if need_atomicity:
            with transaction.atomic(savepoint=False):
                if new_action or new_messages:
                    self._create_action_with_messages(
                        action=new_action,
                        messages=new_messages,
                        context=enqueued_action.context,
                        need_atomicity=False
                    )
                enqueued_action.dequeued_at = timezone.now()
                enqueued_action.save()
        else:
            if new_action or new_messages:
                self._create_action_with_messages(
                    action=new_action,
                    messages=new_messages,
                    context=enqueued_action.context,
                    need_atomicity=False
                )
            enqueued_action.dequeued_at = timezone.now()
            enqueued_action.save()

        return new_action, new_messages

    def _build_final_action(self):
        action = None
        messages = []

        if self._user.get_status() is User.Status.ACTIVE:
            if self._check_app_version(self._user):
                action = self._chat_builder.create_enter_app_action()
            elif self._user.registration_state.update_app_notification_sent_at is None:
                app_install = self._user.app_installs.filter(is_latest=True).first()
                if app_install is None:
                    pass
                elif app_install.get_platform() is AppPlatform.IOS:
                    message_group_id = 'app_update_ios'
                elif app_install.get_platform() is AppPlatform.ANDROID:
                    message_group_id = 'app_update_android'
                else:
                    raise RuntimeError('unreachable: {}'.format(app_install.get_platform()))

                now = timezone.now()
                with transaction.atomic(savepoint=False):
                    action, messages = self.send_message_group(id_=message_group_id)
                    self._user.registration_state.update_app_notification_sent_at = now
                    self._user.registration_state.save()

        return action, messages

    def _check_app_version(self, user):
        return True
        app_install = AppInstall.objects.filter(user=user, is_latest=True).first()
        if app_install is None:
            return False
        if app_install.app_version is None or app_install.app_build is None:
            return False
        return True

    def send_message_group(self, id_):
        message_group = self._script.message_groups[id_]
        messages = self._chat_builder.build_message_group_messages(
            user=self._user,
            message_group=message_group,
        )
        self._create_action_with_messages(action=None, messages=messages)
        return None, messages

    def _create_action_with_messages(self, action, messages, context=None, need_atomicity=True):
        if need_atomicity:
            with transaction.atomic(savepoint=False):
                RegistrationChatMessage.objects.bulk_create(messages)
                if action:
                    RegistrationChatActionResult.objects.create(
                        chat_action_id=action.id,
                        user=self._user,
                        suggested_at=timezone.now(),
                        type=action.get_type().value,
                        context=context,
                    )
        else:
            RegistrationChatMessage.objects.bulk_create(messages)
            if action:
                RegistrationChatActionResult.objects.create(
                    chat_action_id=action.id,
                    user=self._user,
                    suggested_at=timezone.now(),
                    type=action.get_type().value,
                    context=context,
                )

    def _initialize_chat(self):
        new_action, new_messages = self.run_until_next_action(current_action_id=None)
        return new_action, new_messages
