import logging

from .actions import (
    BaseChatAction, CallVerifyChatAction, CreditCardChatAction,
    DriverLicenseChatAction, DriverLicenseBackChatAction, DriverLicenseFrontChatAction,
    EnterAppChatAction, OkChatAction, PassportChatAction,
    PassportBiographicalChatAction, PassportRegistrationChatAction, PassportSelfieChatAction,
)
from .items import (
    BaseChatItem, ImageChatItem, RetryActionChatItem, TextChatItem, TextBlockChatItem,
)


LOGGER = logging.getLogger(__name__)


class RegistrationScript(object):

    def __init__(self, flow, action_groups, message_groups):
        self.flow = flow
        self.action_groups = action_groups
        self.message_groups = message_groups

    @classmethod
    def from_dict(cls, raw_script, context):
        parser = DictRegistrationScriptParser(context=context)
        return parser.parse(raw_script)

    def get_action_group_by_id(self, action_group_id):
        return self.action_groups[action_group_id]

    def get_chat_action_by_id(self, action_group_id):
        try:
            group = self.get_action_group_by_id(action_group_id)
            action = group.action
        except KeyError:
            action = None
        return action

    def get_message_group_by_id(self, message_group_id):
        return self.message_groups[message_group_id]


class FlowItem(object):

    def __init__(self, action_group_id):
        self.action_group_id = action_group_id


class ActionGroup(object):

    def __init__(self, id_, pre_action, action, post_action):
        self.id = id_
        self.pre_action = pre_action
        self.action = action
        self.post_action = post_action


class PreAction(object):

    def __init__(self, chat_items):
        self.chat_items = chat_items


class PostAction(object):

    def __init__(self, chat_items):
        self.chat_items = chat_items


class MessageGroup(object):

    def __init__(self, id_, chat_items):
        self.id = id_
        self.chat_items = chat_items


class DictRegistrationScriptParser(object):

    def __init__(self, context):
        self._context = context

    def parse(self, raw_script):
        flow = []
        for raw_flow_item in raw_script['flow']:
            flow_item = self._parse_flow_item(raw_flow_item)
            flow.append(flow_item)

        action_groups = {}
        for raw_action_group in raw_script['action_groups']:
            action_group = self._parse_action_group(raw_action_group)
            action_groups[action_group.id] = action_group

        message_groups = {}
        for raw_message_group in raw_script['message_groups']:
            message_group = self._parse_message_group(raw_message_group)
            message_groups[message_group.id] = message_group

        script = RegistrationScript(
            flow=flow,
            action_groups=action_groups,
            message_groups=message_groups,
        )

        return script

    def _parse_flow_item(self, raw_flow_item):
        action_group_id = raw_flow_item['action_group_id']
        flow_item = FlowItem(
            action_group_id=action_group_id,
        )
        return flow_item

    def _parse_action_group(self, raw_action_group):
        id_ = raw_action_group['id']
        pre_action = self._parse_pre_action(
            id_=id_,
            raw_pre_action=raw_action_group['pre_action'],
        )
        action = self._parse_action(
            id_=id_,
            raw_action=raw_action_group['action'],
        )
        post_action = self._parse_post_action(
            id_=id_,
            raw_post_action=raw_action_group['post_action'],
        )

        action_group = ActionGroup(
            id_=id_,
            pre_action=pre_action,
            action=action,
            post_action=post_action,
        )

        return action_group

    def _parse_pre_action(self, id_, raw_pre_action):
        chat_items = self._parse_chat_items(id_, raw_pre_action)
        pre_action = PreAction(chat_items=chat_items)
        return pre_action

    def _parse_action(self, id_, raw_action):
        assert len(raw_action) == 1
        raw_action = raw_action[0]

        type_ = BaseChatAction.Type(raw_action['type'])
        params = raw_action['params']

        if type_ is BaseChatAction.Type.CALL_VERIFY:
            action = CallVerifyChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.CREDIT_CARD:
            action = CreditCardChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.DRIVER_LICENSE:
            action = DriverLicenseChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.DRIVER_LICENSE_BACK:
            action = DriverLicenseBackChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.DRIVER_LICENSE_FRONT:
            action = DriverLicenseFrontChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.ENTER_APP:
            action = EnterAppChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.OK:
            action = OkChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.PASSPORT:
            action = PassportChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.PASSPORT_BIOGRAPHICAL:
            action = PassportBiographicalChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.PASSPORT_REGISTRATION:
            action = PassportRegistrationChatAction(id_=id_, text=params['text'])
        elif type_ is BaseChatAction.Type.PASSPORT_SELFIE:
            action = PassportSelfieChatAction(id_=id_, text=params['text'])
        else:
            LOGGER.error('unknown chat action type: %s', type_)
            raise RuntimeError('unreachable: {}'.format(type_))

        return action

    def _parse_post_action(self, id_, raw_post_action):
        chat_items = self._parse_chat_items(id_, raw_post_action)
        post_action = PostAction(chat_items=chat_items)
        return post_action

    def _parse_message_group(self, raw_message_group):
        id_ = raw_message_group['id']
        chat_items = self._parse_chat_items(id_, raw_message_group['chat_items'])
        return MessageGroup(
            id_=id_,
            chat_items=chat_items,
        )

    def _parse_chat_items(self, id_, raw_chat_items):  # pylint: disable=unused-argument
        chat_items = []
        for raw_chat_item in raw_chat_items:
            chat_item = self._parse_chat_item(raw_chat_item)
            chat_items.append(chat_item)
        return chat_items

    def _parse_chat_item(self, raw_chat_item):
        type_ = BaseChatItem.Type(raw_chat_item['type'])
        params = raw_chat_item.get('params', {})
        when = raw_chat_item.get('when')

        if type_ is BaseChatItem.Type.IMAGE:
            chat_item = ImageChatItem(
                url=params['url'],
                on_tap=params.get('on_tap'),
                when=when,
            )
        elif type_ is BaseChatItem.Type.RETRY_ACTION:
            chat_item = RetryActionChatItem(when=when)
        elif type_ is BaseChatItem.Type.TEXT:
            chat_item = TextChatItem(
                message=params['message'],
                on_tap=params.get('on_tap'),
                when=when,
            )
        elif type_ is BaseChatItem.Type.TEXT_BLOCK:
            chat_item = TextBlockChatItem(
                messages=params['messages'],
                when=when,
            )
        else:
            raise RuntimeError('unreachable')

        return chat_item
