import copy
import collections
import datetime
import enum
import logging
import math
import operator
import random

import dateutil.parser
import pymorphy2
from django.db import transaction
from django.db.models import F
from django.utils import timezone

import cars.settings
from cars.core.passport import PassportClient
from cars.core.pusher import BasePusher
from cars.core.util import datetime_helper
from cars.registration.models.chat_action_queue import RegistrationChatActionQueue
from cars.registration.models.chat_action_result import RegistrationChatActionResult
from cars.registration_yang.models.assignment import YangAssignment
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.core.user_profile_updater import UserProfileUpdater
from cars.users.models.user import User
from cars.users.models.user_documents import UserDocument, UserDocumentPhoto
from .chat.manager import ChatManager


LOGGER = logging.getLogger(__name__)


class RegistrationManager(object):

    class AlreadyRegisteredError(Exception):
        pass

    class ChatNotCompletedError(Exception):
        pass

    class DatasyncDataMissingError(Exception):
        pass

    class ForeignDocumentsError(Exception):
        pass

    class NotRejectedError(Exception):
        pass

    class PhotoResubmitInProgressError(Exception):
        pass


    USER_REGISTRATION_STATUSES = {
        User.Status.ONBOARDING,
        User.Status.SCREENING,
    }
    USER_REJECTED_STATUSES = {
        User.Status.BAD_AGE,
        User.Status.BAD_DRIVING_EXPERIENCE,
        User.Status.EXPIRED_DRIVER_LICENSE,
        User.Status.REGISTRATION_FRAUD,
        User.Status.REJECTED,
    }

    def __init__(self, chat_manager, datasync_client, pusher):
        self._chat_manager = chat_manager
        self._datasync = datasync_client
        self._pusher = pusher

    @classmethod
    def from_settings(cls):
        return cls(
            chat_manager=ChatManager.from_settings(),
            datasync_client=DataSyncDocumentsClient.from_settings(),
            pusher=BasePusher.from_settings(),
        )

    def register_all(self):
        self.ingest_yang_assignments()

    def ingest_yang_assignments(self):
        yang_assignments = (
            YangAssignment.objects
            .select_related(
                'passport_biographical',
                'passport_registration',
                'passport_selfie',
                'license_front',
                'license_back',
            )
            .filter(
                processed_at__isnull=False,
                ingested_at__isnull=True,
                license_back__user__registration_state__chat_completed_at__isnull=False,
            )
            .order_by('-processed_at')
        )

        for yang_assignment in yang_assignments:
            try:
                self.ingest_yang_assignment(yang_assignment)
            except Exception:
                LOGGER.exception('failed to ingest yang assignment')
                continue

    def ingest_yang_assignment(self, yang_assignment):
        user = (
            UserDocument.objects
            .select_related('user__registration_state')
            .get(id=yang_assignment.passport_biographical.document_id)
            .user
        )

        if not cars.settings.IS_TESTS:
            UserProfileUpdater(user=user).update_document_hashes()

        try:
            self._ensure_chat_completed(user)
        except self.ChatNotCompletedError:
            LOGGER.info(
                'ready yang assignment %s for user %s with active chat',
                yang_assignment.id,
                user.id,
            )
            return

        passport_data, license_data = self.update_user_data(user=user)

        # Only update personal data for already registered users.
        if user.get_status() not in self.USER_REGISTRATION_STATUSES:
            self.mark_assignment_ingested(yang_assignment)
            return

        if user.uid in cars.settings.REGISTRATION['fake_uids']:
            updater = UserProfileUpdater(user=user)
            with transaction.atomic(savepoint=False):
                updater.update_status(User.Status.ACTIVE)
                yang_assignment.ingested_at = timezone.now()
                yang_assignment.save()
            return

        is_foreign = False
        try:
            photos_to_resubmit = self._get_photos_to_resubmit(yang_assignment)
        except self.ForeignDocumentsError:
            is_foreign = True
            photos_to_resubmit = None

        with transaction.atomic(savepoint=False):
            fraud_status = YangAssignment.Status(yang_assignment.is_fraud)
            if photos_to_resubmit:
                try:
                    all_assignments = (
                        YangAssignment.objects
                        .filter(
                            license_back__user=user,
                            created_at__gte=timezone.now() - datetime.timedelta(days=14),
                        )
                    )
                    has_def_fraud = False
                    for assignment in all_assignments:
                        if str(assignment.is_fraud).lower() == 'definitely_fraud':
                            has_def_fraud = True
                    if has_def_fraud and all_assignments.count() >= 3:
                        self.mark_assignment_ingested(yang_assignment)
                        return
                except Exception:
                    pass

                photo_types = sorted(
                    [p.get_type() for p in photos_to_resubmit],
                    key=operator.attrgetter('value'),
                )
                self.resubmit_photos(user=user, photo_types=photo_types, need_atomicity=False)
            else:
                yang_assignment.is_experimental = True
                self.mark_assignment_ingested(yang_assignment)
                return

            self.mark_assignment_ingested(yang_assignment)

    def make_notifier(self, user):
        notifier = RegistrationManagerUserNotifier(
            user=user,
            chat_manager=self._chat_manager,
            pusher=self._pusher,
        )
        return notifier

    def queue_user_for_screening(self, user):
        LOGGER.info('queueing user for screening: %s', user.id)

        if user.get_status() is User.Status.SCREENING:
            LOGGER.info('user is already in screening: %s', user.id)
            return

        assert user.get_status() is User.Status.ONBOARDING

        with transaction.atomic(savepoint=False):
            user = (
                User.objects
                .select_for_update()
                .get(id=user.id)
            )
            updater = UserProfileUpdater(user=user)
            updater.update_status(User.Status.SCREENING)

    def approve(self, user, force=False):
        if user.get_status() not in self.USER_REGISTRATION_STATUSES:
            raise self.AlreadyRegisteredError('user.registered')

        if not force:
            self._ensure_chat_completed(user)

        LOGGER.info('approving user: %s', user.id)

        with transaction.atomic(savepoint=False):
            user = (
                User.objects
                .select_for_update()
                .get(id=user.id)
            )

            notifier = self.make_notifier(user=user)
            updater = UserProfileUpdater(user=user)

            updater.update_status(User.Status.ACTIVE)
            updater.update_registered_at(registered_at=timezone.now())
            try:
                notifier.notify_approve()
            except notifier.AlreadyNotifiedError:
                LOGGER.exception('user already notified')
            except Exception:
                LOGGER.exception('push service in temporarily down')

        return user

    def approve_rejected(self, user):
        if user.get_status() not in self.USER_REJECTED_STATUSES:
            raise self.NotRejectedError('reject.missing')

        self._ensure_chat_completed(user)

        LOGGER.info('approving user after a reject: %s', user.id)

        with transaction.atomic(savepoint=False):
            user = (
                User.objects
                .select_for_update()
                .get(id=user.id)
            )

            notifier = self.make_notifier(user=user)
            updater = UserProfileUpdater(user=user)

            updater.update_status(User.Status.ACTIVE, force=True)
            updater.update_registered_at(registered_at=timezone.now())

            try:
                notifier.notify_approve_rejected()
            except Exception:
                LOGGER.exception('unable to notify')

        return user

    def reject(self, user):
        if user.get_status() not in self.USER_REGISTRATION_STATUSES:
            raise self.AlreadyRegisteredError('user.registered')

        self._ensure_chat_completed(user)

        LOGGER.info('rejecting user: %s', user.id)

        with transaction.atomic(savepoint=False):
            user = (
                User.objects
                .select_for_update()
                .get(id=user.id)
            )

            notifier = self.make_notifier(user=user)
            updater = UserProfileUpdater(user=user)

            updater.update_status(User.Status.REJECTED)
            updater.update_registered_at(registered_at=timezone.now())

            try:
                notifier.notify_bad_no_reason()
            except Exception:
                LOGGER.exception('unable to notify')

        return user

    def mark_assignment_ingested(self, yang_assignment):
        LOGGER.info('marking assignment %s ingested', yang_assignment.id)
        yang_assignment.ingested_at = timezone.now()
        yang_assignment.save()

    def _ensure_chat_completed(self, user):
        registration_state = user.get_registration_state()
        if registration_state is None or registration_state.chat_completed_at is None:
            raise self.ChatNotCompletedError('chat.incomplete')

    def _get_photos_to_resubmit(self, yang_assignment):
        photos = [
            yang_assignment.passport_biographical,
            yang_assignment.passport_registration,
            yang_assignment.passport_selfie,
            yang_assignment.license_front,
            yang_assignment.license_back,
        ]

        photos_per_status = collections.defaultdict(list)
        for photo in photos:
            photos_per_status[photo.get_verification_status()].append(photo)

        ok_photos = photos_per_status[UserDocumentPhoto.VerificationStatus.OK]

        foreign_photos = photos_per_status[UserDocumentPhoto.VerificationStatus.FOREIGN]

        photos_to_resubmit = (
            photos_per_status[UserDocumentPhoto.VerificationStatus.NEED_INFO] +
            photos_per_status[UserDocumentPhoto.VerificationStatus.DISCARDED] +
            photos_per_status[UserDocumentPhoto.VerificationStatus.NON_LATIN] +
            photos_per_status[UserDocumentPhoto.VerificationStatus.UNRECOGNIZABLE]
        )

        if foreign_photos:
            raise self.ForeignDocumentsError

        if not photos_to_resubmit:
            if len(ok_photos) != len(photos):
                raise RuntimeError('some photos are in unknown state for assignment {}'.format(yang_assignment.id))

        return photos_to_resubmit

    def resubmit_photos(self, user, photo_types, need_atomicity=True):
        if self._is_resubmit_in_progress(user):
            raise self.PhotoResubmitInProgressError

        self._ensure_chat_completed(user)

        notifier = self.make_notifier(user=user)
        n_photos_to_resubmit = len(photo_types)

        if n_photos_to_resubmit == 0:
            raise RuntimeError('unexpected 0 photos to resubmit for user {}'.format(user.id))
        elif n_photos_to_resubmit == 1:
            photo_type = photo_types[0]
            self._resubmit_one_photo(photo_type, user=user, notifier=notifier)
        elif n_photos_to_resubmit == 2:
            photo_type1, photo_type2 = photo_types
            self._resubmit_two_photos(photo_type1, photo_type2, user=user, notifier=notifier, need_atomicity=need_atomicity)
        else:
            self._resubmit_two_plus_photos(photo_types, user=user, notifier=notifier, need_atomicity=need_atomicity)

    def _is_resubmit_in_progress(self, user):
        resubmit_chat_actions = [
            'resubmit_only_license_back',
            'resubmit_only_license_front',
            'resubmit_only_passport_bio',
            'resubmit_only_passport_reg',
            'resubmit_only_passport_selfie',
            'resubmit_2_from_license_back',
            'resubmit_2_from_license_front',
            'resubmit_2_from_passport_bio',
            'resubmit_2_from_passport_reg',
            'resubmit_2_from_passport_selfie',
            'cont_resubmit_2p_license_back',
            'cont_resubmit_2p_license_front',
            'cont_resubmit_2p_passport_bio',
            'cont_resubmit_2p_passport_reg',
            'cont_resubmit_2p_passport_selfie',
        ]

        actions = (
            RegistrationChatActionResult.objects
            .filter(
                user=user,
                completed_at__isnull=True,
                chat_action_id__in=resubmit_chat_actions,
            )
        )
        if actions.exists():
            return True

        queue = (
            RegistrationChatActionQueue.objects
            .filter(
                user=user,
                dequeued_at__isnull=True,
                chat_action_id__in=resubmit_chat_actions,
            )
        )
        if queue.exists():
            return True

        return False

    def _resubmit_one_photo(self, photo_type, user, notifier):
        chat_action_id = {
            UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'resubmit_only_license_back',
            UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'resubmit_only_license_front',
            UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'resubmit_only_passport_bio',
            UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'resubmit_only_passport_reg',
            UserDocumentPhoto.Type.PASSPORT_SELFIE: 'resubmit_only_passport_selfie',
        }[photo_type]
        self._enqueue_chat_action(user=user, chat_action_id=chat_action_id)
        notifier.notify_resubmit_single_photo()

    def _resubmit_two_photos(self, photo_type1, photo_type2, user, notifier, need_atomicity=True):
        start_chat_action_id = {
            UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'resubmit_2_from_license_back',
            UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'resubmit_2_from_license_front',
            UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'resubmit_2_from_passport_bio',
            UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'resubmit_2_from_passport_reg',
            UserDocumentPhoto.Type.PASSPORT_SELFIE: 'resubmit_2_from_passport_selfie',
        }[photo_type1]

        photo_names = []
        for photo_type in [photo_type1, photo_type2]:
            photo_name = {
                UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'оборотной стороны водительского удостоверения',  # pylint: disable=line-too-long
                UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'лицевой стороны водительского удостоверения',  # pylint: disable=line-too-long
                UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'разворота паспорта с вашим фото',
                UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'разворота паспорта с регистрацией',
                UserDocumentPhoto.Type.PASSPORT_SELFIE: 'селфи с паспортом',
            }[photo_type]
            photo_names.append(photo_name)
        resubmit_two_photos_message = 'Нужно переделать фотографии {} и {}.'.format(*photo_names)

        finish_chat_action_id = {
            UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'cont_resubmit_2p_license_back',
            UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'cont_resubmit_2p_license_front',
            UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'cont_resubmit_2p_passport_bio',
            UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'cont_resubmit_2p_passport_reg',
            UserDocumentPhoto.Type.PASSPORT_SELFIE: 'cont_resubmit_2p_passport_selfie',
        }[photo_type2]

        if need_atomicity:
            with transaction.atomic(savepoint=False):
                self._enqueue_chat_action(
                    user=user,
                    chat_action_id=start_chat_action_id,
                    context={
                        'resubmit_two_photos_message': resubmit_two_photos_message,
                    },
                )
                self._enqueue_chat_action(
                    user=user,
                    chat_action_id=finish_chat_action_id,
                    context={
                        'cont_resubmit_2p_photos_final': True,
                    },
                )
                notifier.notify_resubmit_multiple_photos()
        else:
            self._enqueue_chat_action(
                user=user,
                chat_action_id=start_chat_action_id,
                context={
                    'resubmit_two_photos_message': resubmit_two_photos_message,
                },
            )
            self._enqueue_chat_action(
                user=user,
                chat_action_id=finish_chat_action_id,
                context={
                    'cont_resubmit_2p_photos_final': True,
                },
            )
            notifier.notify_resubmit_multiple_photos()

    def _resubmit_two_plus_photos(self, photo_types, user, notifier, need_atomicity=True):
        assert len(photo_types) > 2

        start_photo_type = photo_types[0]
        start_chat_action_id = {
            UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'resubmit_2_from_license_back',
            UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'resubmit_2_from_license_front',
            UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'resubmit_2_from_passport_bio',
            UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'resubmit_2_from_passport_reg',
            UserDocumentPhoto.Type.PASSPORT_SELFIE: 'resubmit_2_from_passport_selfie',
        }[start_photo_type]

        photo_names = []
        for photo_type in photo_types:
            photo_name = {
                UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'оборотной стороны водительского удостоверения',  # pylint: disable=line-too-long
                UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'лицевой стороны водительского удостоверения',  # pylint: disable=line-too-long
                UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'разворота паспорта с вашим фото',
                UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'разворота паспорта с регистрацией',
                UserDocumentPhoto.Type.PASSPORT_SELFIE: 'селфи с паспортом',
            }[photo_type]
            photo_names.append(photo_name)

        resubmit_two_plus_photos_message = (
            'Нужно переделать фотографии: {} и {}.'
            .format(
                ', '.join(photo_names[:-1]),
                photo_names[-1],
            )
        )

        if need_atomicity:
            with transaction.atomic(savepoint=False):
                self._enqueue_chat_action(
                    user=user,
                    chat_action_id=start_chat_action_id,
                    context={
                        'resubmit_two_photos_message': resubmit_two_plus_photos_message,
                    },
                )

                for i, photo_type in enumerate(photo_types[1:], 1):
                    is_final = i == len(photo_types) - 1
                    chat_action_id = {
                        UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'cont_resubmit_2p_license_back',
                        UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'cont_resubmit_2p_license_front',
                        UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'cont_resubmit_2p_passport_bio',
                        UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'cont_resubmit_2p_passport_reg',
                        UserDocumentPhoto.Type.PASSPORT_SELFIE: 'cont_resubmit_2p_passport_selfie',
                    }[photo_type]
                    self._enqueue_chat_action(
                        user=user,
                        chat_action_id=chat_action_id,
                        context={
                            'cont_resubmit_2p_photos_final': is_final,
                        },
                    )

                notifier.notify_resubmit_multiple_photos()
        else:
            self._enqueue_chat_action(
                user=user,
                chat_action_id=start_chat_action_id,
                context={
                    'resubmit_two_photos_message': resubmit_two_plus_photos_message,
                },
            )

            for i, photo_type in enumerate(photo_types[1:], 1):
                is_final = i == len(photo_types) - 1
                chat_action_id = {
                    UserDocumentPhoto.Type.DRIVER_LICENSE_BACK: 'cont_resubmit_2p_license_back',
                    UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT: 'cont_resubmit_2p_license_front',
                    UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL: 'cont_resubmit_2p_passport_bio',
                    UserDocumentPhoto.Type.PASSPORT_REGISTRATION: 'cont_resubmit_2p_passport_reg',
                    UserDocumentPhoto.Type.PASSPORT_SELFIE: 'cont_resubmit_2p_passport_selfie',
                }[photo_type]
                self._enqueue_chat_action(
                    user=user,
                    chat_action_id=chat_action_id,
                    context={
                        'cont_resubmit_2p_photos_final': is_final,
                    },
                )

            notifier.notify_resubmit_multiple_photos()

    def _handle_definitely_fraud_user(self, user):
        self.queue_user_for_screening(user)

    def _handle_maybe_fraud_user(self, user):
        self.queue_user_for_screening(user)

    def _handle_not_fraud_user(self, user, passport_data, license_data):
        with transaction.atomic(savepoint=False):
            if 'birth_date' in passport_data:
                birth_date = dateutil.parser.parse(passport_data['birth_date']).date()
            elif 'birth_date' in license_data:
                birth_date = dateutil.parser.parse(license_data['birth_date']).date()
            else:
                UserProfileUpdater(user=user).add_tags(['no_birth_date', ])
                self.queue_user_for_screening(user)
                return

            b_start_date, b_end_date = self._get_category_b_span(license_data)

            age_check_result = self._check_user_age(
                birth_date,
            )
            driving_experience_check = self._check_user_driving_experience(
                b_start_date,
                b_end_date,
            )
            reliability_check = self._check_user_reliability(user, passport_data, license_data)

            if not reliability_check:
                UserProfileUpdater(user=user).add_tags(['reliability_chk_failed', ])
                self.queue_user_for_screening(user)
            elif birth_date and datetime.date.today() - birth_date > datetime.timedelta(days=55 * 356):
                self.reject(user=user)
            elif not age_check_result.ok:
                LOGGER.info('age check not ok for user %s', user.id)
                self.queue_user_for_screening(user)
            elif driving_experience_check.status is DrivingExperienceCheckResult.Status.LOW:
                LOGGER.info('driving experience is low for user %s', user.id)
                self.queue_user_for_screening(user)
            elif driving_experience_check.status is DrivingExperienceCheckResult.Status.EXPIRED:
                LOGGER.info('driving experience is expired for user %s', user.id)
                UserProfileUpdater(user=user).add_tags(['expired_license_check', ])
                self.queue_user_for_screening(user)
            elif driving_experience_check.status is DrivingExperienceCheckResult.Status.UNKNOWN:
                LOGGER.info('driving experience is unknown for user %s', user.id)
                UserProfileUpdater(user=user).add_tags(['invalid_license_cat_or_date', ])
                self.queue_user_for_screening(user)
            else:
                try:
                    self.approve(user=user)
                except UserProfileUpdater.StatusChangeError:
                    LOGGER.info('status can not be changed')
                    self.reject(user)

    def _handle_foreign_user(self, user):
        LOGGER.info('foreign user %s', user.id)
        self.queue_user_for_screening(user)

    def update_user_data(self, user):
        passport_data = self._datasync.get_passport_unverified(user.uid, user.passport_ds_revision)
        license_data = self._datasync.get_license_unverified(user.uid, user.driving_license_ds_revision)
        updater = UserProfileUpdater(user=user)

        if passport_data and ('doc_value' in passport_data) and ('first_name' in passport_data) and ('last_name' in passport_data) and ('middle_name' in passport_data):
            updater.update_passport(
                passport_number=passport_data['doc_value'],
                first_name=passport_data['first_name'],
                last_name=passport_data['last_name'],
                patronymic_name=passport_data['middle_name'],
            )
        else:
            LOGGER.warning('missing passport data for user %s', user.id)

        if license_data and ('number' in license_data) and ('first_name' in license_data) and ('last_name' in license_data) and ('middle_name' in license_data):
            updater.update_driver_license(
                driver_license_number=license_data['number'],
                first_name=license_data.get('first_name', ''),
                last_name=license_data.get('last_name', ''),
                patronymic_name=license_data.get('middle_name', ''),
            )
        else:
            LOGGER.warning('missing license data for user %s', user.id)

        return passport_data, license_data

    def _get_category_b_span(self, license_data):
        # has a copycat in ban manager; no reason to refactor now
        if 'B' not in license_data['categories'].upper() or 'issue_date' not in license_data:
            return None, None

        issue_date = dateutil.parser.parse(license_data['issue_date']).date()

        if 'experience_from' in license_data:
            start_date_str = license_data['experience_from']
        elif 'categories_b_valid_from_date' in license_data:
            start_date_str = license_data['categories_b_valid_from_date']
        else:
            # Old format driver license.
            if issue_date > datetime.date(2014, 1, 1):
                masked_license_data = {k: '***' for k in license_data}
                LOGGER.error('invalid new style driver license: %s', masked_license_data)
                start_date_str = None
            else:
                start_date_str = license_data['issue_date']

        if start_date_str is None:
            start_date = None
        else:
            start_date = dateutil.parser.parse(start_date_str).date()

        end_date_str = license_data.get('categories_b_valid_to_date')
        if end_date_str is None:
            end_date = datetime_helper.add_years(issue_date, years=10)
        else:
            end_date = dateutil.parser.parse(end_date_str).date()

        # Infer start_date/end_date from each other if any one of them is missing.
        if start_date and not end_date:
            end_date = datetime_helper.add_years(start_date, years=10)
        if not start_date and end_date:
            start_date = datetime_helper.add_years(end_date, years=-10)

        return start_date, end_date

    def _check_user_age(self, birth_date):
        reference_date = max(datetime.date.today(), cars.settings.REGISTRATION['start_date'])
        delta = reference_date - birth_date
        days_to_ok = abs(min(delta.days - cars.settings.REGISTRATION['min_age'].days, 0))
        ok = days_to_ok == 0
        return AgeCheckResult(ok=ok, days_to_ok=days_to_ok)

    def _check_user_driving_experience(self, b_start_date, b_end_date):
        reference_date = max(datetime.date.today(), cars.settings.REGISTRATION['start_date'])
        days_to_ok = None

        if b_start_date is None or b_end_date is None:
            status = DrivingExperienceCheckResult.Status.UNKNOWN
        elif b_end_date < reference_date:
            status = DrivingExperienceCheckResult.Status.EXPIRED
        else:
            delta = reference_date - b_start_date
            days_to_ok = abs(
                min(
                    delta.days - cars.settings.REGISTRATION['min_driving_experience'].days,
                    0,
                ),
            )
            if days_to_ok == 0:
                status = DrivingExperienceCheckResult.Status.OK
            else:
                status = DrivingExperienceCheckResult.Status.LOW

        return DrivingExperienceCheckResult(status=status, days_to_ok=days_to_ok)

    def _check_user_reliability(self, user, passport_data, driving_license_data):
        passport_data = passport_data or {}
        driving_license_data = driving_license_data or {}

        is_all_countries_ok = True
        has_country_filled = False
        has_untrusted_countries = False
        if 'biographical_country' in passport_data:
            has_country_filled = True
            if passport_data['biographical_country'] in cars.settings.REGISTRATION['untrusted_countries']:
                LOGGER.info('Untrusted country met: {}'.format(passport_data['biographical_country']))
                has_untrusted_countries = True
            if passport_data['biographical_country'] not in cars.settings.REGISTRATION['trusted_countries']:
                LOGGER.info('Mismatch: {}'.format(passport_data['biographical_country']))
                is_all_countries_ok = False
        if 'registration_country' in passport_data:
            has_country_filled = True
            if passport_data['registration_country'] in cars.settings.REGISTRATION['untrusted_countries']:
                LOGGER.info('Untrusted country met: {}'.format(passport_data['registration_country']))
                has_untrusted_countries = True
            if passport_data['registration_country'] not in cars.settings.REGISTRATION['trusted_countries']:
                LOGGER.info('Mismatch: {}'.format(passport_data['registration_country']))
                is_all_countries_ok = False
        if 'front_country' in driving_license_data:
            has_country_filled = True
            if driving_license_data['front_country'] in cars.settings.REGISTRATION['untrusted_countries']:
                LOGGER.info('Untrusted country met: {}'.format(driving_license_data['front_country']))
                has_untrusted_countries = True
            if driving_license_data['front_country'] not in cars.settings.REGISTRATION['trusted_countries']:
                LOGGER.info('Mismatch: {}'.format(driving_license_data['front_country']))
                is_all_countries_ok = False
        if 'back_country' in driving_license_data:
            has_country_filled = True
            if driving_license_data['back_country'] in cars.settings.REGISTRATION['untrusted_countries']:
                LOGGER.info('Untrusted country met: {}'.format(driving_license_data['back_country']))
                has_untrusted_countries = True
            if driving_license_data['back_country'] not in cars.settings.REGISTRATION['trusted_countries']:
                LOGGER.info('Mismatch: {}'.format(driving_license_data['back_country']))
                is_all_countries_ok = False

        if has_untrusted_countries:
            is_ok, vr = self._check_user(user, has_untrusted_country=True)
            LOGGER.info(
                'EX2-FOREIGNERS: user {} has untrusted countries, and her is_ok value is {} / seeds: {} / acc.age: {}'
                .format(
                    'https://carsharing.yandex-team.ru/#/clients/' + str(user.id) + '/docs',
                    is_ok,
                    str(vr.seeds),
                    str(vr.account_age)
                )
            )
            return is_ok
        elif not has_untrusted_countries and not is_all_countries_ok:
            is_ok, vr = self._check_user(user, has_untrusted_country=False)
            LOGGER.info(
                'EX2-FOREIGNERS: user {} has unknown countries, and her is_ok value is {} / seeds: {} / acc.age: {}'
                .format(
                    'https://carsharing.yandex-team.ru/#/clients/' + str(user.id) + '/docs',
                    is_ok,
                    str(vr.seeds),
                    str(vr.account_age)
                )
            )
            return is_ok

        if not is_all_countries_ok:
            LOGGER.info('Not all countries were OK, not pass.')
            return False

        if has_country_filled:
            LOGGER.info('All countries were OK, pass.')
            return True

        if not has_country_filled and len(passport_data) > 5 and len(driving_license_data) > 5:
            LOGGER.info('No country data and filled fields. Pass.')
            return True

        LOGGER.info('Has country filled: {}'.format(has_country_filled))
        LOGGER.info(
            'Len passport_data: {} Len driving_license_data: {}'.format(
                len(passport_data),
                len(driving_license_data)
            )
        )
        return False

    def _check_user(self, user, has_untrusted_country):
        if has_untrusted_country:
            LOGGER.info('User has untrusted countries.')
        else:
            LOGGER.info('User has no untrusted countries, but also has unknown countries.')

        verifier = PassportClient.from_settings()
        db_fields = copy.deepcopy(cars.settings.BLACKBOX['db_fields_basic'])
        for db_field_extra in cars.settings.BLACKBOX['suid_weights'].keys():
            db_fields.append(db_field_extra)
        try:
            verification_result = verifier.get_account_info(uid=user.uid, db_fields=db_fields)
        except Exception:
            LOGGER.exception('Unable to verify user.')
            return False
        LOGGER.info('Account age: {}'.format(verification_result.account_age))
        LOGGER.info('Number of subscriptions: {}'.format(verification_result.num_subscriptions_weighted))
        if has_untrusted_country:
            return (
                verification_result.account_age > datetime.timedelta(days=365) and
                verification_result.num_subscriptions_weighted >= 2
            ), verification_result
        else:
            return (
                verification_result.account_age > datetime.timedelta(days=365) or
                verification_result.num_subscriptions_weighted >= 2
            ), verification_result

    def _enqueue_chat_action(self, user, chat_action_id, context=None):
        session = self._chat_manager.make_session(user=user)
        session.enqueue_chat_action(
            chat_action_id=chat_action_id,
            context=context,
        )
        # Force chat evaluation to update the current chat_action_id.
        if user.get_registration_state().chat_action_id is None:
            session.get_chat_state()

    def _delete_yang_assignment(self, yang_assignment):
        LOGGER.info('bad assignment %s', yang_assignment.id)
        # yang_assignment.delete()


AgeCheckResult = collections.namedtuple(
    'AgeCheckResult',
    ['ok', 'days_to_ok'],
)


class DrivingExperienceCheckResult(object):

    class Status(enum.Enum):
        EXPIRED = 'expired'
        LOW = 'low'
        OK = 'ok'
        UNKNOWN = 'unknown'

    def __init__(self, status, days_to_ok):
        self.status = status
        self.days_to_ok = days_to_ok


class RegistrationManagerUserNotifier(object):

    class AlreadyNotifiedError(Exception):
        pass

    class ChatMessageGroups(object):
        FINISH_BAD_AGE = 'finish_bad_age'
        FINISH_BAD_DRIVING_EXPERIENCE = 'finish_bad_driving_experience'
        FINISH_BAD_NO_REASON = 'finish_bad_no_reason'
        FINISH_OK = 'finish_ok'
        FINISH_OK_SORRY = 'finish_ok_sorry'

    class PushMessages(object):
        APPROVE = 'Ура, вы зарегистрированы в Яндекс.Драйве!'
        REJECT = 'К сожалению, мы не можем зарегистрировать вас'

    def __init__(self, user, chat_manager, pusher):
        self._user = user
        self._user_app_install = None
        self._chat_manager = chat_manager
        self._chat_session = None
        self._pusher = pusher

    def _send_chat_message_group(self, message_group_id, context=None):
        session = self._chat_manager.make_session(user=self._user, context=context)
        session.send_message_group(id_=message_group_id)

    def _send_push_notification(self, message):
        try:
            self._pusher.send(uid=self._user.uid, message=message)
        except Exception:
            LOGGER.exception('push service is down')

    def _send_ok_push_notification(self):
        self._send_push_notification(message=self.PushMessages.APPROVE)

    def _send_bad_push_notification(self):
        self._send_push_notification(message=self.PushMessages.REJECT)

    def notify_approve(self):
        registration_state = self._user.get_registration_state()
        if registration_state.outcome_notification_sent_at is not None:
            raise self.AlreadyNotifiedError(
                self._user.id,
                registration_state.outcome_notification_sent_at,
            )

        registration_state.outcome_notification_sent_at = timezone.now()

        with transaction.atomic(savepoint=False):
            self.notify_ok()
            registration_state.save()

    def notify_approve_rejected(self):
        registration_state = self._user.get_registration_state()
        registration_state.outcome_notification_sent_at = timezone.now()
        with transaction.atomic(savepoint=False):
            self.notify_ok_sorry()
            registration_state.save()

    def notify_ok(self):
        self._send_chat_message_group(
            message_group_id=self.ChatMessageGroups.FINISH_OK,
            context={
                'user': self._user,
            },
        )
        self._send_ok_push_notification()

    def notify_ok_sorry(self):
        self._send_chat_message_group(
            message_group_id=self.ChatMessageGroups.FINISH_OK_SORRY,
            context={
                'user': self._user,
            },
        )
        self._send_ok_push_notification()

    def notify_bad_age(self):
        self._send_chat_message_group(
            message_group_id=self.ChatMessageGroups.FINISH_BAD_AGE,
            context={
                'user': self._user,
            },
        )
        self._send_bad_push_notification()

    def notify_bad_driving_experience(self, days_to_ok):
        time_to_ok_driving_experience = self._format_time_to_ok_driving_experience_string(
            days_to_ok=days_to_ok,
        )
        self._send_chat_message_group(
            message_group_id=self.ChatMessageGroups.FINISH_BAD_DRIVING_EXPERIENCE,
            context={
                'user': self._user,
                'time_to_ok_driving_experience': time_to_ok_driving_experience,
            },
        )
        self._send_bad_push_notification()

    def _format_time_to_ok_driving_experience_string(self, days_to_ok):
        months_to_ok = int(math.ceil(days_to_ok / 30))
        morph = pymorphy2.MorphAnalyzer()
        month_word = morph.parse('месяц')[0].inflect({'accs'})
        months_str = month_word.make_agree_with_number(months_to_ok).word
        return '{} {}'.format(months_to_ok, months_str)

    def notify_bad_no_reason(self):
        self._send_chat_message_group(
            message_group_id=self.ChatMessageGroups.FINISH_BAD_NO_REASON,
        )
        self._send_bad_push_notification()

    def notify_resubmit_single_photo(self):
        self._send_push_notification(message='Пожалуйста, сделайте заново одну фотографию')

    def notify_resubmit_multiple_photos(self):
        self._send_push_notification(message='Пожалуйста, сделайте заново несколько фотографий')
