import datetime
import logging
from decimal import Decimal

from django.db import transaction
from django.db.models import Sum
from django.utils import timezone
import pytz

import cars.settings
from cars.billing.core.bonus_account_manager import BonusAccountManager
from cars.core.util import make_yt_client
from ...models import RegistrationTaxiRide
from .users_helper import UsersHelper


LOGGER = logging.getLogger(__name__)

TAXI_CASHBACK_PERCENT = cars.settings.REGISTRATION['taxi_cashback']['cashback_proportion']
MAX_CASHBACK_VALUE = cars.settings.REGISTRATION['taxi_cashback']['max_cashback_value']
CASHBACK_ACCRUAL_END_TIME = cars.settings.REGISTRATION['taxi_cashback']['accrual_end_time']


class RidesHelper:

    def __init__(self):
        self._yt_client = make_yt_client('data')
        self._users_helper = UsersHelper()

    def update_ride_records(self):
        raw_rides = self._get_rides_raw_data()
        phone_to_latest_ride = self._get_latest_ride_for_each_user()
        phone_to_user = self._users_helper.get_phone_to_user_mapping()

        new_rides = 0
        users_with_changed_cashback = set()

        with transaction.atomic():
            for raw_ride in raw_rides:
                new_ride = self._ride_from_taxi_format(
                    raw_ride,
                    phone_to_latest_ride,
                    phone_to_user,
                )
                if not new_ride:
                    continue

                LOGGER.info('New taxi ride: %s', new_ride.id)
                new_ride.save()
                new_rides += 1

                users_with_changed_cashback.add(new_ride.user)

            LOGGER.info('New rides: %d\nModifying user records.', new_rides)
            for user in users_with_changed_cashback:
                total_cashback = (
                    RegistrationTaxiRide.objects
                    .filter(user=user)
                    .aggregate(Sum('cost'))['cost__sum']
                ) * Decimal(TAXI_CASHBACK_PERCENT)

                if total_cashback > MAX_CASHBACK_VALUE:
                    total_cashback = Decimal(MAX_CASHBACK_VALUE)

                user.registration_state.total_taxi_cashback = total_cashback
                user.registration_state.save()

                try:
                    bonus_account_manager = BonusAccountManager.from_user(user)
                    bonus_account_manager.update_registration_taxi_cashback_earned(total_cashback)
                except Exception:
                    LOGGER.exception('bonus account update failed')

        LOGGER.info('Modified %d user registration states.', len(users_with_changed_cashback))

    def _get_existing_ride_ids(self):
        existing_ride_ids = set()
        ride_objects = RegistrationTaxiRide.objects.all()
        for ride in ride_objects:
            existing_ride_ids.add(ride.id)
        return existing_ride_ids

    def _get_latest_ride_for_each_user(self):
        """
        Get the latest counted ride for each user.

        Why works: https://docs.djangoproject.com/en/1.11/ref/models/querysets/#distinct

        ...For example, SELECT DISTINCT ON (a) gives you the first row for each
        value in column a. If you don't specify an order, you'll get some
        arbitrary row...
        """
        latest_rides = (
            RegistrationTaxiRide.objects
            .order_by(
                'user__phone',
                '-ride_utc_time'
            )
            .distinct('user__phone')
            .select_related('user')
        )
        phone_to_latest_ride = {}
        for ride in latest_rides:
            user = ride.user
            if user.phone is None:
                LOGGER.error('Found a ride for a user with incorrect phone. '
                             'E-mail: %s , ride time: %s', str(user.email), str(user.phone))
                continue
            phone_to_latest_ride[user.phone.as_e164] = ride.ride_utc_time
        return phone_to_latest_ride

    def _get_rides_raw_data(self):
        return self._yt_client.read_table(
            cars.settings.REGISTRATION['taxi_cashback']['rides_table_path']
        )

    def _restore_datetime(self, dt_str):
        result = datetime.datetime.strptime(
            dt_str,
            '%Y-%m-%d %H:%M:%S',
        )
        return timezone.make_aware(result, timezone=pytz.UTC)

    def _ride_from_taxi_format(self, raw_ride, phone_to_latest_ride, phone_to_user):
        ride_id = raw_ride['order_id']
        ride_utc_time = self._restore_datetime(raw_ride['utc_order_due_dttm'])

        if ride_utc_time > CASHBACK_ACCRUAL_END_TIME:
            LOGGER.info('Cashback accual has ended.')
            return None

        phone_number = raw_ride['number']
        cost = raw_ride['order_cost']

        if cost < 0.1:  # If cost is 'very low'
            LOGGER.error('Faulty ride data: %s', str(raw_ride))
            return None

        if phone_number not in phone_to_user:
            LOGGER.error(
                'Found a ride for a user which is not yet active or unregistered. '
                'Phone: %s', phone_number
            )
            return None

        user_had_rides = phone_number in phone_to_latest_ride
        if user_had_rides and ride_utc_time <= phone_to_latest_ride[phone_number]:
            return None

        user = phone_to_user[phone_number]
        if ride_utc_time < user.registration_state.chat_completed_at:
            LOGGER.error('User completed chat after the ride %s.', ride_id)
            return None

        return RegistrationTaxiRide(
            id=ride_id,
            user=user,
            ride_utc_time=ride_utc_time,
            cost=Decimal(cost),
        )
