import logging

import cars.settings
from cars.core.util import make_yt_client
from cars.users.models import User


LOGGER = logging.getLogger(__name__)


class UsersHelper:

    def __init__(self):
        self._yt_client = make_yt_client('data')

    def get_phone_to_user_mapping(self):
        registered_users = self._get_registered_users()
        mapping = {}
        for user in registered_users:
            mapping[user.phone] = user
        return mapping

    def update_users_phones_at_yt(self):
        users = self._get_registered_users()

        table_data = []
        for user in users:
            if user.phone is None:
                LOGGER.error('Incorrect phone of user %s: %s', str(user.email), str(user.phone))
                continue
            user_entry = {
                'phone': user.phone.as_e164,
                'registered_at': self._get_min_date(
                    user.registration_state.chat_completed_at,
                    user.registered_at
                ).isoformat(),
            }
            table_data.append(user_entry)

        LOGGER.info("Uploading %d users' phones.", len(table_data))

        self._yt_client.write_table(
            cars.settings.REGISTRATION['taxi_cashback']['phones_table_path'],
            table_data,
        )

    def _get_min_date(self, chat_completion_date, registration_date):
        if not chat_completion_date:
            return registration_date
        if not registration_date:
            return chat_completion_date
        return min(chat_completion_date, registration_date)

    def _get_registered_users(self):
        return (
            User.objects
            .filter(
                registration_state__isnull=False,
                status=User.Status.ACTIVE.value,
            )
            .select_related('bonus_account', 'registration_state')
            .filter(registration_state__chat_completed_at__isnull=False)
        )
