import datetime
import enum
import logging

from django.db import transaction
from django.utils import timezone

import cars.settings
from cars.core.util import import_class
from ..core.solomon import SOLOMON
from ..models.verification_call import RegistrationVerificationCall
from .chat.manager import ChatManager


LOGGER = logging.getLogger(__name__)


class RegistrationVerificationCallResolver(object):

    class SessionStatus(enum.Enum):
        INPROGRESS = 0
        OK = 1
        UNANSWER = 2
        OTHER = 3

    def __init__(self, chat_manager, octopus_client, call_timeout, call_check_lag, solomon_client):
        self._chat = chat_manager
        self._octopus = octopus_client
        self._call_timeout = datetime.timedelta(seconds=call_timeout)
        self._call_check_lag = datetime.timedelta(seconds=call_check_lag)
        self._solomon_client = solomon_client

    @classmethod
    def from_settings(cls):
        octopus_client = import_class(cars.settings.OCTOPUS['client_class']).from_settings()
        return cls(
            chat_manager=ChatManager.from_settings(),
            octopus_client=octopus_client,
            call_timeout=cars.settings.REGISTRATION['octopus']['call_timeout'],
            call_check_lag=cars.settings.REGISTRATION['octopus']['call_check_lag'],
            solomon_client=SOLOMON,
        )

    def _get_duration_solomon_sensor(self):
        return 'registration.verification_call_resolver.call.duration'

    def _get_octopus_session_fail_solomon_sensor(self):
        return 'registration.octopus_session_fail'

    def _get_status_solomon_sensor(self, status):
        return 'registration.verification_call_resolver.call.status.{}'.format(status)

    def resolve_all(self):
        min_submitted_at = timezone.now() - self._call_check_lag  # do not check calls submitted right now

        calls = (
            RegistrationVerificationCall.objects
            .filter(
                status=RegistrationVerificationCall.Status.INPROGRESS.value,
                submitted_at__lte=min_submitted_at,
            )
            .select_related('user')
            .order_by('submitted_at')
        )

        for call in calls:
            self.resolve(call)

    def resolve(self, call):
        if not call.user.phone:
            call.status = self.SessionStatus.OK.value
            call.resolved_at = timezone.now()
            with transaction.atomic(savepoint=False):
                call.save()
                self._chat.complete_chat_action(call.chat_action_result, need_atomicity=False)

        if call.status != RegistrationVerificationCall.Status.INPROGRESS.value:
            LOGGER.warning('call %s already resolved, skipping', call.id)
            return

        LOGGER.info('resolving octopus session %s', call.session_id)

        session_status = self.get_session_status(call.session_id)

        call_status = None
        sensor_status = None

        now = timezone.now()
        call_duration = now - call.submitted_at

        if session_status is self.SessionStatus.INPROGRESS:
            if call_duration > self._call_timeout:
                LOGGER.info('octopus session %s is timed out', call.session_id)
                call_status = RegistrationVerificationCall.Status.ERROR
                sensor_status = 'timeout'
            else:
                LOGGER.info('octopus session %s is in progress', call.session_id)
        elif session_status is self.SessionStatus.OK:
            call_status = RegistrationVerificationCall.Status.OK
            sensor_status = 'ok'
        elif session_status is self.SessionStatus.UNANSWER:
            call_status = RegistrationVerificationCall.Status.NO_PICK_UP
            sensor_status = 'unanswer'
        elif session_status is self.SessionStatus.OTHER:
            call_status = RegistrationVerificationCall.Status.ERROR
            sensor_status = 'other'
        else:
            sensor_status = 'error'
            raise RuntimeError('unreachable: {}'.format(session_status))

        if call_status is not None:
            LOGGER.info('octopus session %s status is %s', call.session_id, call_status.name)
            call.status = call_status.value
            call.resolved_at = timezone.now()
            with transaction.atomic(savepoint=False):
                call.save()
                self._chat.complete_chat_action(call.chat_action_result, need_atomicity=False)

            self._solomon_client.set_value(
                self._get_duration_solomon_sensor(),
                call_duration.total_seconds(),
            )

        if sensor_status:
            self._solomon_client.set_value(self._get_status_solomon_sensor(sensor_status), 1)

    def get_session_status(self, session_id):
        return self.SessionStatus.OK

        try:
            session_log = self._octopus.get_session_log(session_id)
        except Exception:  # octopus fails - validate
            LOGGER.exception('Failed to get octopus session log.')
            try:
                self._solomon_client.set_value(
                    self._get_octopus_session_fail_solomon_sensor(),
                    1
                )
            except Exception:
                LOGGER.exception('Failed to send solomon data')

            return self.SessionStatus.OK

        sections_per_type = {section['action_type']: section for section in session_log}

        if 'ORIGINATE' not in sections_per_type:
            return self.SessionStatus.INPROGRESS

        if sections_per_type.get('PHASE_END', {}).get('action_args', {}).get('state') == 'ERROR':
            LOGGER.error('octopus error for session %s:\n%s', session_id, session_log)
            return self.SessionStatus.OTHER

        state_section = sections_per_type.get('SESSION_DUMP_STATE')
        if state_section is None:
            return self.SessionStatus.INPROGRESS

        if 'PLAY_PROMPT' not in sections_per_type:
            return self.SessionStatus.UNANSWER

        try:
            if state_section['action_args']['state']['variables']['vm_messages']:
                return self.SessionStatus.OK
        except Exception:
            LOGGER.exception('unknown state for octopus session %s:\n%s', session_id, session_log)

        LOGGER.error('failed to categorize octopus session %s:\n%s', session_id, session_log)

        return self.SessionStatus.OTHER
