import time
import datetime
import logging
from decimal import Decimal

from django.db.models import Max, Sum
from django.utils import timezone

import cars.settings
from cars.core.daemons import CarsharingDaemon
from cars.registration.models import RegistrationTaxiRide
from cars.registration_yang.core.assignment import AssignmentManager
from cars.registration_yang.models import YangAssignment
from cars.users.models import RegistrationState, User, UserDocumentPhoto
from cars.registration.core.registration_manager import RegistrationManager


LOGGER = logging.getLogger(__name__)

WAITING_MONITORING_BUCKETS = [
    datetime.timedelta(minutes=10),
    datetime.timedelta(minutes=20),
    datetime.timedelta(minutes=30),
    datetime.timedelta(hours=1),
    datetime.timedelta(hours=2),
    datetime.timedelta(hours=3),
    datetime.timedelta(hours=6),
    datetime.timedelta(hours=9),
    datetime.timedelta(hours=12),
    datetime.timedelta(days=1),
    datetime.timedelta(days=2),
]


class RegistrationMonitoringDaemon(CarsharingDaemon):

    tick_interval = '*/5 * * * *'

    def get_distributed_lock_relative_path(self):
        return 'registration/locks/monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'daemons.registration_monitoring'

    def get_solomon_service(self):
        return cars.settings.REGISTRATION['solomon']['service']

    def _monitor_total_cashbacked_rides(self):
        """
        How many rides participated in cashback promo.
        """
        counted_rides = RegistrationTaxiRide.objects.count()
        self.solomon.set_value('registration.taxi_cashback.rides_counted', counted_rides)

    def _monitor_maximal_cashback_value(self):
        """
        What is the maximal value of cashback.
        If greater than 2000 rubles, alert instantly.
        """
        max_cashback = (
            RegistrationState.objects
            .all()
            .aggregate(Max('total_taxi_cashback'))['total_taxi_cashback__max']
        )
        self.solomon.set_value('registration.taxi_cashback.max_value', float(max_cashback))

    def _monitor_cashbacked_with_zero_cost(self):
        """
        What is the number of rides with cost 0.
        If greater than zero, alert instantly.
        """
        rides_cost_zero = RegistrationTaxiRide.objects.filter(cost=Decimal(0.0)).count()
        self.solomon.set_value('registration.taxi_cashback.zero_cost_rides', rides_cost_zero)

    def _monitor_users_with_nonzero_cashback(self):
        """
        How many users have non-zero value of cashback.
        """
        nonzero_cashback_users = RegistrationState.objects.filter(
            total_taxi_cashback__gt=Decimal(0.0)
        ).count()
        self.solomon.set_value(
            'registration.taxi_cashback.nonzero_cb_users',
            nonzero_cashback_users
        )

    def _monitor_total_cashback_value(self):
        """
        Total bonus value.
        """
        total_cashback = (
            RegistrationState.objects
            .all()
            .aggregate(Sum('total_taxi_cashback'))['total_taxi_cashback__sum']
        )
        self.solomon.set_value('registration.taxi_cashback.total_value', float(total_cashback))

    def _monitor_cashback(self):
        self._monitor_users_with_nonzero_cashback()
        self._monitor_cashbacked_with_zero_cost()
        self._monitor_total_cashbacked_rides()
        self._monitor_maximal_cashback_value()
        self._monitor_total_cashback_value()

    def _monitor_decision_waiting_times(self):
        waiting_users_states = (
            RegistrationState.objects
            .filter(
                chat_completed_at__isnull=False,
                chat_action_id__isnull=True,
                user__status__in=[
                    x.value for x in RegistrationManager.USER_REGISTRATION_STATUSES
                ],
            )
        )
        current_time = timezone.now()
        bucket_values = [0] * (len(WAITING_MONITORING_BUCKETS) + 1)
        for state in waiting_users_states:
            time_waiting = current_time - state.chat_completed_at
            for bucket_idx, bucket_max_time in enumerate(WAITING_MONITORING_BUCKETS):
                if bucket_max_time > time_waiting:
                    bucket_values[bucket_idx] += 1
                    break
            else:
                bucket_values[-1] += 1

        for index, value in enumerate(bucket_values):
            if index + 1 < len(bucket_values):
                sensor_name = 'registration.waiting_times.below_{}_sec'.format(
                    int(WAITING_MONITORING_BUCKETS[index].total_seconds())
                )
            else:
                sensor_name = 'registration.waiting_times.below_inf_sec'
            self.solomon.set_value(sensor_name, value)

    def _monitor_assignments_too_long_waiting(self):
        value = (
            YangAssignment.objects
            .filter(
                created_at__lte=timezone.now() - datetime.timedelta(hours=12),
                processed_at__isnull=True
            )
            .count()
        )

        self.solomon.set_value('registration.waiting_times.too_long_waiting', value)

    def _monitor_not_created_assignments(self):
        dummy_manager = AssignmentManager(None, None, None, None)
        value = dummy_manager.create_new_assignments(
            fake=True,
            not_count_newest=True
        )
        self.solomon.set_value('registration.not_created_assignments', value)

    def _monitor_screening_docs_fine(self):
        stuck_assignments = (
            YangAssignment.objects
            .filter(
                ingested_at__isnull=False,
                license_back__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_front__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_biographical__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_registration__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_selfie__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_back__user__status=User.Status.SCREENING.value,
            )
            .select_related('license_back')
        )

        stuck_users = set()
        for assignment in stuck_assignments:
            stuck_users.add(assignment.license_back.user_id)

        self.solomon.set_value('registration.stuck_in_status.screening', len(stuck_users))

    def _monitor_onboarding_docs_fine(self):
        stuck_assignments_ingested = (
            YangAssignment.objects
            .filter(
                ingested_at__isnull=False,
                license_back__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_front__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_biographical__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_registration__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_selfie__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_back__user__status=User.Status.ONBOARDING.value,
            )
        )

        stuck_users_ingested = set()
        for assignment in stuck_assignments_ingested:
            stuck_users_ingested.add(assignment.license_back.user_id)

        stuck_assignments_processed = (
            YangAssignment.objects
            .filter(
                processed_at__isnull=False,
                ingested_at__isnull=True,
                license_back__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_front__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_biographical__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_registration__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                passport_selfie__verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                license_back__user__status=User.Status.ONBOARDING.value,
            )
        )

        stuck_users_processed = set()
        for assignment in stuck_assignments_processed:
            if assignment.license_back.user_id not in stuck_users_ingested:
                stuck_users_processed.add(assignment.license_back.user_id)

        self.solomon.set_value('registration.stuck_in_status.onboarding.ingested', len(stuck_users_ingested))
        self.solomon.set_value('registration.stuck_in_status.onboarding.processed', len(stuck_users_processed))

    def _do_tick(self):
        self._monitor_assignments_too_long_waiting()
        self._monitor_not_created_assignments()
        self._monitor_screening_docs_fine()
        self._monitor_onboarding_docs_fine()
